<?php

namespace Drupal\webform_address_autocomplete\Plugin\WebformAddressProvider;

use Drupal\Component\Utility\Xss;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Utility\Error;
use Drupal\webform_address_autocomplete\Plugin\WebformAddressProviderBase;
use GuzzleHttp\Exception\RequestException;

/**
 * Defines a PostCh plugin for webform_address_autocomplete.
 *
 * @WebformAddressProvider(
 *   id = "post_ch",
 *   label = @Translation("Post CH"),
 * )
 */
class PostCh extends WebformAddressProviderBase {

  /**
   * {@inheritDoc}
   */
  public function defaultConfiguration() {
    return parent::defaultConfiguration() + [
      'endpoint' => '',
      'username' => '',
      'password' => '',
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['endpoint'] = [
      '#type' => 'url',
      '#title' => $this->t('post.ch API endpoint'),
      '#default_value' => $this->configuration['endpoint'],
    ];
    $form['username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('post.ch API username'),
      '#default_value' => $this->configuration['username'],
    ];
    $form['password'] = [
      '#type' => 'password',
      '#title' => $this->t('post.ch API password'),
      '#default_value' => $this->configuration['password'],
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritDoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $configuration = $this->getConfiguration();

    $configuration['endpoint'] = $form_state->getValue('endpoint');
    $configuration['username'] = $form_state->getValue('username');
    $configuration['password'] = $form_state->getValue('password');

    $this->setConfiguration($configuration);
  }

  /**
   * {@inheritDoc}
   */
  public function processQuery($string) {
    $addresses = $this->prepareRequest($string);
    $results = [];

    foreach ($addresses as $address) {
      $street_name = Xss::filter($address->StreetName);
      $house_number = Xss::filter($address->HouseNo);
      $zip_code = Xss::filter($address->ZipCode);
      $town_name = Xss::filter($address->TownName);

      $results[] = [
        'street_name' => $street_name . " " . $house_number,
        'zip_code' => $zip_code,
        'town_name' => $town_name,
        'label' => $street_name . " " . $house_number . " " . $zip_code . " " . $town_name,
      ];
    }

    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function prepareRequest($string) {
    $request = [
      'request' => [
        'ONRP' => 0,
        'ZipCode' => '',
        'ZipAddition' => '',
        'TownName' => '',
        'STRID' => 0,
        'StreetName' => $string,
        'HouseKey' => 0,
        'HouseNo' => '',
        'HouseNoAddition' => '',
      ],
      'zipOrderMode' => 0,
      'zipFilterMode' => 0,
    ];

    // If last entered word starts with number, let's guess it's house number
    // ie: Schultheissenstrasse 2b.
    $pieces = explode(' ', $string);
    $pos_number = array_pop($pieces);

    if (!empty($pieces) && is_numeric($pos_number)) {
      $request['request']['StreetName'] = implode(' ', $pieces);
      $request['request']['HouseNo'] = $pos_number;
    }

    $results = $this->request($request);

    // Sometimes guessing may be wrong, as numbers could be part of streetname
    // ie: Avenue 14-Avril
    // do fallback here.
    if (empty($results) && !empty($request['request']['HouseNo'])) {
      $request['request']['StreetName'] = $string;
      $request['request']['HouseNo'] = '';
      $results = $this->request($request);
    }

    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function request($request) {
    try {
      $response = $this->client->post(
        $this->configuration['endpoint'],
        [
          'auth' => [
            $this->configuration['username'],
            $this->configuration['password'],
          ],
          'body' => json_encode($request),
          'headers' => [
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
          ],
        ]
      );
      $content = json_decode($response->getBody()->getContents());
      $results = $content->QueryAutoComplete4Result->AutoCompleteResult;
      // Limit the number of results to 10.
      $results = array_slice($results, 0, 10);
    }
    catch (RequestException $e) {
      Error::logException($this->getLogger('webform_address_autocomplete'), $e);
      $results = [];
    }

    return $results;
  }

}
