<?php

namespace Drupal\webform_background_color\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\Entity\Webform;

/**
 * Config form for selecting webforms to enable background color option.
 */
class WebformBackgroundColorConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'webform_background_color_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['webform_background_color.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('webform_background_color.settings');
    $webforms = Webform::loadMultiple();
    $options = [];

    foreach ($webforms as $id => $webform) {
      $options[$id] = $webform->label();
    }

    $selection_mode = $config->get('selection_mode') ?? 'multiple';

    $form['selection_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Selection mode'),
      '#options' => [
        'single' => $this->t('Single webform'),
        'multiple' => $this->t('Multiple webforms'),
      ],
      '#default_value' => $selection_mode,
    ];

    // Single dropdown.
    $form['enabled_webform'] = [
      '#type' => 'select',
      '#title' => $this->t('Select one webform'),
      '#options' => $options,
      '#default_value' => $config->get('enabled_webform') ?? '',
      '#states' => [
        'visible' => [
          ':input[name="selection_mode"]' => ['value' => 'single'],
        ],
      ],
    ];

    // Multi-select dropdown.
    $form['enabled_webforms'] = [
      '#type' => 'select',
      '#title' => $this->t('Select multiple webforms'),
      '#options' => $options,
      '#default_value' => $config->get('enabled_webforms') ?? [],
      '#multiple' => TRUE,
      '#size' => 5,
      '#states' => [
        'visible' => [
          ':input[name="selection_mode"]' => ['value' => 'multiple'],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $selection_mode = $form_state->getValue('selection_mode');

    $this->config('webform_background_color.settings')
      ->set('selection_mode', $selection_mode);

    if ($selection_mode === 'single') {
      $enabled_webform = $form_state->getValue('enabled_webform');
      $this->config('webform_background_color.settings')
        ->set('enabled_webforms', [$enabled_webform])
        ->set('enabled_webform', $enabled_webform);
    }
    else {
      $enabled_webforms = array_filter($form_state->getValue('enabled_webforms'));
      $this->config('webform_background_color.settings')
        ->set('enabled_webforms', $enabled_webforms)
        ->clear('enabled_webform');
    }

    $this->config('webform_background_color.settings')->save();

    parent::submitForm($form, $form_state);
  }

}
