<?php

namespace Drupal\webform_background_color\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\Entity\Webform;

/**
 * Config form for selecting webforms to enable background color option.
 */
class WebformBackgroundColorConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'webform_background_color_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['webform_background_color.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('webform_background_color.settings');
    $webforms = Webform::loadMultiple();
    $options = [];

    foreach ($webforms as $id => $webform) {
      $options[$id] = $webform->label();
    }

    $selection_mode = $config->get('selection_mode') ?? 'multiple';
    $default_values = $config->get('enabled_webforms') ?? [];

    $form['selection_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow multiple selections'),
      '#default_value' => $selection_mode === 'multiple',
    ];

    $form['enabled_webforms'] = [
      '#type' => 'select',
      '#title' => $this->t('Select Webform(s)'),
      '#options' => $options,
      '#default_value' => $default_values,
      '#multiple' => TRUE,
      '#size' => 5,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $is_multiple = $form_state->getValue('selection_mode');
    $selection_mode = $is_multiple ? 'multiple' : 'single';
    $selected = array_filter($form_state->getValue('enabled_webforms'));

    // If single mode, store only the first selected webform.
    if (!$is_multiple && !empty($selected)) {
      $selected = [reset($selected)];
    }

    $this->config('webform_background_color.settings')
      ->set('selection_mode', $selection_mode)
      ->set('enabled_webforms', $selected)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
