<?php

/**
 * @file
 * Builds placeholder replacement tokens for webform_booking elements.
 */

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;

/**
 * Implements hook_token_info().
 *
 * Declares custom tokens for webform booking elements.
 */
function webform_booking_token_info() {
  $date_config_url = Url::fromRoute('entity.date_format.collection')->toString();

  $booking = [];
  $booking['booking'] = [
    'name' => t('Webform booking information'),
    'dynamic' => TRUE,
    'description' => Markup::create(t('Date, time or number of slots of a booking.') .
    '<br /><strong>' . t('Learn about booking tokens') . '</strong><br />' .
    t("To output individual elements, replace the '?' with …") . '<br />' .
    '<ul>' .
    '<li>
      <b>element_key</b><br />' .
      t("Default setting. Displays the raw date and time.") .
    '</li>' .
    '<li>
      <b>element_key:date</b><br />' .
      t("Displays the date formatted in the built in <i>html_date</i> format.") .
    '</li>' .
    '<li>
      <b>element_key:date:*</b><br />' .
      t("To select a date format, replace the '*' with the key of a <a href=\"@datetimeformats_link\">custom date and time format</a>.", [
        "@datetimeformats_link" => $date_config_url,
      ]) .
    '</li>' .
    '<li>
      <b>element_key:time</b><br />' .
      t("Displays the booking time formatted as 'H:i'.") .
    '</li>' .
    '<li>
      <b>element_key:slots</b><br />' .
      t("Displays the number of slots booked.") .
    '</li>' .
    '<li>
      <b>element_key:cancel_link</b><br />' .
      t("Display a link to cancel booking(s) on a submission. You have to specify an element_key but the link is the same for all webform booking fields on a form.") .
    '</li>' .
    '</ul>'),
  ];

  $tokens = [];
  $tokens['webform_submission'] = $booking;
  return ['types' => [], 'tokens' => $tokens];

}

/**
 * Implements hook_tokens().
 *
 * Generates tokens for date, time and slot information that can be used in
 * confirmation pages or emails generated by the Webform module.
 */
function webform_booking_tokens($type, $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata) {
  $token_service = \Drupal::token();
  $replacements = [];

  if ($type === 'webform_submission' && !empty($data['webform_submission'])) {
    /** @var \Drupal\webform\WebformSubmissionInterface $webform_submission */
    $webform_submission = $data['webform_submission'];

    // Iterate all webform booking tokens.
    $value_tokens = $token_service->findWithPrefix($tokens, 'booking');
    if (empty($value_tokens)) {
      return $replacements;
    }

    foreach ($value_tokens as $value_token => $original) {
      $keys = explode(':', $value_token);
      // Extract name of booking form element.
      $element_key = array_shift($keys);
      // Extract info to display. Either 'slots', 'date' or 'cancel link'.
      $element_part = array_shift($keys);

      // Generate a cancel link.
      if ($element_part === 'cancel_link') {
        $replacements[$original] = _webform_booking_create_cancel_link($webform_submission);
        continue;
      }

      // Extract slot date, time and count, eg. "2025-02-17 11:40|1".
      $element_value = $webform_submission->getElementData($element_key);

      // Validate stored booking info. Hide token on error.
      $pattern = '/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}\|\d+$/';
      if (empty($element_value) || preg_match($pattern, $element_value) !== 1) {
        // Invalid field data pattern.
        $replacements[$original] = NULL;
        continue;
      }
      $slot_data = explode('|', $element_value);
      // Extract date.
      $slot_datetime = new DrupalDateTime($slot_data[0]);
      if ($slot_datetime->hasErrors()) {
        // Invalid date contained in field data.
        $replacements[$original] = NULL;
        continue;
      }
      // Extract number of slots.
      $slot_count = empty($slot_data[1]) ? 0 : $slot_data[1];

      if ($slot_count === 0) {
        $replacements[$original] = t('Cancelled');
        continue;
      }

      switch ($element_part) {
        // Display number of slots booked.
        case 'slots':
          $value = $slot_count ?? 0;
          break;

        // Display booked time only.
        case 'time':
          $value = _webform_booking_format_date($slot_datetime, 'custom', 'H:i');
          break;

        // Display booked date only with optional date format.
        case 'date':
          $element_format = !empty($keys[0]) ? array_shift($keys) : 'html_date';
          $value = _webform_booking_format_date($slot_datetime, $element_format);
          break;

        // Display raw value, ie. date and time.
        default:
          $value = $slot_data[0];
      }

      if ($value !== NULL) {
        $replacements[$original] = $value;
      }
    }

  }

  return $replacements;
}

/**
 * Create a link to cancel the associated submission.
 *
 * Creates a cancellation link based on booking information and the site's
 * hash salt.
 *
 * @param \Drupal\webform\WebformSubmissionInterface $webform_submission
 *   The webform submission object.
 *
 * @return mixed
 *   A link to cancel the booking or FALSE on error.
 */
function _webform_booking_create_cancel_link($webform_submission) {
  try {
    /** @var \Drupal\webform\WebformRequestInterface $request_handler */
    $request_handler = \Drupal::service('webform.request');
    $route_parameters = $request_handler->getRouteParameters($webform_submission, $webform_submission->getSourceEntity());
    $url = Url::fromRoute('webform_booking.cancel_booking_confirm', $route_parameters);

    $options = $url->setAbsolute()->getOptions();
    $options['query']['token'] = $webform_submission->getToken();
    return $url->setOptions($options)->toString();
  }
  catch (\Exception $e) {
    return FALSE;
  }
}
