<?php

namespace Drupal\webform_campaignmonitor\Plugin\WebformHandler;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\Plugin\WebformHandlerBase;
use Drupal\webform\WebformSubmissionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form submission to Campaignmonitor handler.
 *
 * @WebformHandler(
 *   id = "campaignmonitor",
 *   label = @Translation("CampaignMonitor"),
 *   category = @Translation("CampaignMonitor"),
 *   description = @Translation("Sends a form submission to a CampaignMonitor list."),
 *   cardinality = \Drupal\webform\Plugin\WebformHandlerInterface::CARDINALITY_UNLIMITED,
 *   results = \Drupal\webform\Plugin\WebformHandlerInterface::RESULTS_PROCESSED,
 *   submission = \Drupal\webform\Plugin\WebformHandlerInterface::SUBMISSION_REQUIRED,
 *   tokens = TRUE,
 * )
 */
class WebformCampaignMonitorHandler extends WebformHandlerBase {

  /**
   * The token manager.
   *
   * @var \Drupal\webform\WebformTranslationManagerInterface
   */
  protected $tokenManager;

  /**
   * The CampaignMonitor subscription manager.
   *
   * @var Drupal\campaignmonitor\CampaignMonitorSubscriptionManager
   */
  protected $cmSubscriptionManager;

  /**
   * The campaignmonitor manager.
   *
   * @var \Drupal\campaignmonitor\CampaignMonitorManager
   */
  protected $cmManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->tokenManager = $container->get('webform.token_manager');
    $instance->cmSubscriptionManager = $container->get('campaignmonitor.subscription_manager');
    $instance->cmManager = $container->get('campaignmonitor.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getSummary() {
    $lists = $this->cmManager->getLists();
    $list_name = (isset($this->configuration['list']) && !empty($lists[$this->configuration['list']]))
      ? $lists[$this->configuration['list']]['name'] : '';
    $formatted_list = $this->t(
      '<strong>@list: </strong>@list_name',
      [
        '@list' => 'List',
        '@list_name' => $list_name,
      ]
    );
    return [
      '#markup' => $formatted_list,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'list' => '',
      'email' => '',
      'double_optin' => TRUE,
      'mergevars' => '',
      'interest_groups' => [],
      'control' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $lists = $this->cmManager->getLists();

    $options = [];
    $options[''] = $this->t('- Select an option -');
    foreach ($lists as $key => $list) {
      $options[$key] = $list['name'];
    }

    $form['campaignmonitor'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Campaign monitor settings'),
    ];

    $form['campaignmonitor']['list'] = [
      '#type' => 'webform_select_other',
      '#title' => $this->t('List'),
      '#required' => TRUE,
      '#default_value' => $this->configuration['list'],
      '#options' => $options,
      '#description' => $this->t('Select the list you want to send this submission to. Alternatively, you can also use the Other field for token replacement.'),
    ];

    $fields = $this->getWebform()->getElementsDecoded();
    $options = [];
    $options[''] = $this->t('- Select an option -');
    foreach ($fields as $field_name => $field) {
      if ($field['#type'] == 'email') {
        $options[$field_name] = $field['#title'];
      }
    }

    $form['campaignmonitor']['email'] = [
      '#type' => 'select',
      '#title' => $this->t('Email field'),
      '#required' => TRUE,
      '#default_value' => $this->configuration['email'],
      '#options' => $options,
    ];

    $options = [];
    $options[''] = $this->t('- Select an option -');
    foreach ($fields as $field_name => $field) {
      if ($field['#type'] == 'checkbox') {
        $options[$field_name] = $field['#title'];
      }
    }

    $form['campaignmonitor']['control'] = [
      '#type' => 'select',
      '#title' => $this->t('Control field'),
      '#default_value' => $this->configuration['control'],
      '#options' => $options,
      '#description' => $this->t('An optional control checkbox field to prevent unwanted subscriptions.'),
    ];

    $form['campaignmonitor']['mergevars'] = [
      '#type' => 'webform_codemirror',
      '#mode' => 'yaml',
      '#title' => $this->t('Merge vars'),
      '#default_value' => $this->configuration['mergevars'],
      '#description' => $this->t('Enter the mergevars that will be sent to campaignmonitor, each line a <em>mergevar: "value\"</em>. You may use tokens.'),
    ];

    $form['campaignmonitor']['double_optin'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Double opt-in'),
      '#default_value' => $this->configuration['double_optin'],
    ];

    $form['campaignmonitor']['token_tree_link'] = $this->tokenManager->buildTreeLink();

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    $values = $form_state->getValues();
    foreach ($this->configuration as $name => $value) {
      if (isset($values['campaignmonitor'][$name])) {
        $this->configuration[$name] = $values['campaignmonitor'][$name];
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(WebformSubmissionInterface $webform_submission, $update = TRUE) {
    // If update, do nothing.
    if ($update) {
      return;
    }

    $fields = $webform_submission->toArray(TRUE);

    // If there's a checkbox configured, check for its value.
    if (!empty($this->configuration['control']) && empty($fields['data'][$this->configuration['control']])) {
      return;
    }

    // Replace tokens.
    $configuration = $this->tokenManager->replace($this->configuration, $webform_submission);

    $email = $fields['data'][$configuration['email']];

    $mergevars = Yaml::decode($configuration['mergevars']);
    $name = $mergevars['name'];

    $this->cmSubscriptionManager->userSubscribe($configuration['list'], $email, $name, $mergevars, $configuration['interest_groups'], $configuration['double_optin']);
  }

}
