<?php

declare(strict_types=1);

namespace Drupal\webform_error_field\Hook;

use Drupal\Component\Utility\Html;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Render\Element;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Template\Attribute;
use Drupal\webform\Element\WebformHtmlEditor;

/**
 * Hook implementations for webform_error_field.
 */
class WebformErrorFieldHooks {

  use StringTranslationTrait;

  /**
   * Implements hook_theme().
   */
  #[Hook('theme')]
  public function theme(): array {
    return [
      'webform_element_validation_message' => [
        'variables' => [
          'validation_message' => NULL,
          'attributes' => [],
        ],
      ],
    ];
  }

  /**
   * Implements hook_preprocess_HOOK().
   *
   * Prepares variables for webform validation message templates.
   *
   * Default template: webform-element-validation-message.html.twig.
   *
   * @param array $variables
   *   An associative array containing the following key:
   *   - element: The webform element.
   */
  #[Hook('preprocess_webform_element_validation_message')]
  public function preprocessWebformElement(array &$variables): void {
    if (!is_array($variables['validation_message'])) {
      $variables['validation_message'] = [
        '#markup' => $variables['validation_message'],
        '#allowed_tags' => WebformHtmlEditor::getAllowedTags(),
      ];
    }

    $variables['attributes'] = new Attribute($variables['attributes']);

    // Make sure there is a unique id.
    if (empty($variables['id'])) {
      $variables['id'] = Html::getUniqueId('webform-element-validation-message');
    }

    // Make sure attributes id is set.
    if (!isset($variables['attributes']['id'])) {
      $variables['attributes']['id'] = $variables['id'];
    }
  }

  /**
   * Implements hook_form_alter().
   */
  #[Hook('form_alter')]
  public function formAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    // Only process webform submissions.
    if (!str_starts_with($form_id, 'webform_submission_')) {
      return;
    }

    // Add validation handler that will show validation message elements.
    // This needs to be at the END of validation handlers.
    $form['#validate'][] = [self::class, 'validateShowMessages'];
  }

  /**
   * Validation handler to show validation message elements.
   *
   * This runs AFTER all validation, so we can modify the form to show messages.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function validateShowMessages(array &$form, FormStateInterface $form_state): void {
    // Only show validation messages if there are validation errors.
    if ($form_state::hasAnyErrors()) {
      // Recursively find and show all validation message elements.
      self::showValidationMessagesRecursive($form, $form_state);
    }
  }

  /**
   * Recursively show validation message elements.
   *
   * @param array $element
   *   The form element to process.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  private static function showValidationMessagesRecursive(array &$element, FormStateInterface $form_state): void {
    // Check if this is a validation message element.
    if (isset($element['#type']) && $element['#type'] === 'webform_validation_message') {
      $element['#access'] = TRUE;
    }

    // Recursively process child elements.
    foreach (Element::children($element) as $key) {
      self::showValidationMessagesRecursive($element[$key], $form_state);
    }
  }

}
