<?php

declare(strict_types=1);

namespace Drupal\webform_error_field\Plugin\WebformElement;

use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\Plugin\WebformElement\WebformMarkupBase;
use Drupal\webform\Plugin\WebformElementDisplayOnInterface;
use Drupal\webform\WebformSubmissionInterface;

/**
 * Provides a 'validation_message' element.
 *
 * This element displays rich text (CKEditor content) only when the form
 * has validation errors.
 *
 * @WebformElement(
 *   id = "webform_validation_message",
 *   label = @Translation("Validation Message"),
 *   description = @Translation("Provides a message that appears only when the form has validation errors."),
 *   category = @Translation("Markup elements"),
 * )
 */
class WebformValidationMessage extends WebformMarkupBase {

  /**
   * {@inheritdoc}
   */
  protected function defineDefaultProperties() {
    return [
      'validation_message' => '',
      'attributes' => [],
      // Markup settings.
      'display_on' => WebformElementDisplayOnInterface::DISPLAY_ON_FORM,
    ] + $this->defineDefaultBaseProperties();
  }

  /**
   * {@inheritdoc}
   */
  public function prepare(array &$element, ?WebformSubmissionInterface $webform_submission = NULL): void {
    parent::prepare($element, $webform_submission);

    // Initially hide the element - it will be shown via hook_form_alter()
    // #after_build callback if there are validation errors.
    $element['#access'] = FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    $form['markup']['#title'] = $this->t('Validation message settings');

    $form['markup']['validation_message'] = [
      '#type' => 'webform_html_editor',
      '#title' => $this->t('Validation message'),
      '#description' => $this->t('The message to display when the form has validation errors. You can use HTML and rich text formatting.'),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function preview() {
    return parent::preview() + [
      '#validation_message' => '<p><strong>Please review the errors below</strong> and correct them before submitting the form.</p>',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isInput(array $element) {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function isContainer(array $element) {
    return FALSE;
  }

}
