<?php

namespace Drupal\webform_gohighlevel\Plugin\WebformHandler;

use Drupal\webform\Plugin\WebformHandlerBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\WebformSubmissionInterface;
use GuzzleHttp\Exception\RequestException;

/**
 * Sends Webform submissions to GoHighLevel.
 *
 * @WebformHandler(
 *   id = "gohighlevel_handler",
 *   label = @Translation("GoHighLevel Handler"),
 *   category = @Translation("CRM"),
 *   description = @Translation("Send webform submissions to GoHighLevel."),
 *   cardinality = \Drupal\webform\Plugin\WebformHandlerInterface::CARDINALITY_SINGLE,
 *   results = TRUE,
 * )
 */
class GoHighLevelHandler extends WebformHandlerBase {

  /**
   * Default configuration.
   */
  public function defaultConfiguration() {
    return [
      'api_key' => '',
      'field_mapping' => [],
      'custom_tag' => '',
	  'dynamic_tag_field' => '',
      'source' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * Build the configuration form.
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $webform = $this->getWebform();
    $elements = $webform->getElementsInitializedFlattenedAndHasValue();

    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('Enter the GoHighLevel API key.'),
      '#default_value' => $this->configuration['api_key'] ?? '',
      '#maxlength' => 256,
      '#required' => TRUE,
    ];

    $form['custom_tag'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom Tag'),
      '#description' => $this->t('Enter a custom tag for this webform submission.'),
      '#default_value' => $this->configuration['custom_tag'] ?? '',
      '#maxlength' => 256,
    ];

    // MOVED UP: We need the list of webform fields before we can create the select list.
    $webform_options = [];
    foreach ($elements as $key => $element) {
      $webform_options[$key] = $element['#title'] ?? $key;
    }

    // MOVED UP: This field now appears directly after the 'Custom Tag' field.
    $form['dynamic_tag_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Select Tag Field'),
      '#description' => $this->t('Optionally, select a webform field (like a select list) to use as a dynamic tag. This tag will be added in addition to the Custom Tag above.'),
      '#options' => $webform_options,
      '#default_value' => $this->configuration['dynamic_tag_field'] ?? '',
      '#empty_option' => $this->t('- None -'),
    ];

    // -- END OF REORDERED SECTION --

    $form['source'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Source'),
      '#description' => $this->t('Will be overwritten by the selected value in the webform if available.'),
      '#default_value' => $this->configuration['source'] ?? '',
      '#maxlength' => 128,
    ];

    $form['field_mapping'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Field Mapping'),
      '#description' => $this->t('Map your webform fields to GoHighLevel fields.'),
    ];

    $gohighlevel_fields = [
      'firstName' => $this->t('First Name'),
      'lastName' => $this->t('Last Name'),
      'email' => $this->t('Email'),
      'phone' => $this->t('Phone'),
      'notes' => $this->t('Message'),
      'address1' => $this->t('Street Address'),
      'city' => $this->t('City'),
      'state' => $this->t('State'),
      'postalCode' => $this->t('Postal Code'),
    ];

    foreach ($gohighlevel_fields as $gohighlevel_field => $label) {
      $form['field_mapping'][$gohighlevel_field] = [
        '#type' => 'select',
        '#title' => $label,
        '#options' => $webform_options,
        '#default_value' => $this->configuration['field_mapping'][$gohighlevel_field] ?? '',
        '#empty_option' => $this->t('- Select a Webform field -'),
      ];
    }

    return $form;
  }

  /**
   * Submit configuration form.
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['api_key'] = $form_state->getValue('api_key');
    $this->configuration['custom_tag'] = $form_state->getValue('custom_tag');
	$this->configuration['dynamic_tag_field'] = $form_state->getValue('dynamic_tag_field');
    $this->configuration['source'] = $form_state->getValue('source');
    $this->configuration['field_mapping'] = $form_state->getValue('field_mapping');
    parent::submitConfigurationForm($form, $form_state);
  }
  /**
   * Submit form handler.
   */
  public function submitForm(array &$form, FormStateInterface $form_state, WebformSubmissionInterface $webform_submission) {
    $api_key = trim($this->configuration['api_key'] ?? '');
    $source_default = trim($this->configuration['source'] ?? '');

    $field_mapping = $this->configuration['field_mapping'];
    $data = $webform_submission->getData();

    $payload = [];

    foreach ($field_mapping as $gohighlevel_field => $webform_field) {
      if (!empty($webform_field) && isset($data[$webform_field])) {
        if ($gohighlevel_field === 'notes') {
          $payload['notes'] = $data[$webform_field];
        }
        else {
          $payload[$gohighlevel_field] = $data[$webform_field];
        }
      }
    }

    // START: Modified Tag Handling Logic.
    $tags = [];
    $static_tag = trim($this->configuration['custom_tag'] ?? '');
    $dynamic_tag_field_key = $this->configuration['dynamic_tag_field'] ?? NULL;

    // 1. Add the static tag from the handler configuration if it exists.
    if (!empty($static_tag)) {
      $tags[] = $static_tag;
    }

    // 2. Add the dynamic tag from the webform submission if the field is configured and has a value.
    if (!empty($dynamic_tag_field_key) && !empty($data[$dynamic_tag_field_key])) {
      $tags[] = $data[$dynamic_tag_field_key];
    }
    
    // 3. Add the tags to the payload if we have any.
    if (!empty($tags)) {
      $payload['tags'] = $tags;
    }
    // END: Modified Tag Handling Logic.

    // Set source from webform field if present.
    $source_field_key = 'where_did_you_find_me';
    if (!empty($data[$source_field_key])) {
      $payload['source'] = $data[$source_field_key];
    }
    elseif (!empty($source_default)) {
      $payload['source'] = $source_default;
    }

    $url = 'https://rest.gohighlevel.com/v1/contacts/';

    try {
      \Drupal::httpClient()->post($url, [
        'headers' => [
          'Authorization' => 'Bearer ' . $api_key,
          'Content-Type' => 'application/json',
        ],
        'body' => json_encode($payload),
      ]);
    }
    catch (RequestException $e) {
      \Drupal::logger('webform_gohighlevel')->error('API error: @message', ['@message' => $e->getMessage()]);
    }
  }
}