<?php

namespace Drupal\webform_googleform\Plugin\WebformHandler;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Utility\Error;
use Drupal\webform\Plugin\WebformElement\Email;
use Drupal\webform\Plugin\WebformElementManagerInterface;
use Drupal\webform\Plugin\WebformHandlerBase;
use Drupal\webform\WebformSubmissionInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Psr7\Query;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides the webform_googleform hander.
 *
 * @WebformHandler(
 *   id = "webform_googleform",
 *   label = @Translation("Google Form"),
 *   category = @Translation("External"),
 *   description = @Translation("Pass-through submissions to a Google form."),
 *   cardinality = \Drupal\webform\Plugin\WebformHandlerInterface::CARDINALITY_UNLIMITED,
 *   results = \Drupal\webform\Plugin\WebformHandlerInterface::RESULTS_PROCESSED,
 *   submission = \Drupal\webform\Plugin\WebformHandlerInterface::SUBMISSION_OPTIONAL,
 * )
 */
final class GoogleFormHandler extends WebformHandlerBase implements ContainerFactoryPluginInterface {

  /**
   * The http client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $client;

  /**
   * The webform element manager.
   *
   * @var \Drupal\webform\Plugin\WebformElementManagerInterface
   */
  protected WebformElementManagerInterface $elementManager;

  /**
   * The current request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * Initialization method.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request_stack service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config.factory service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger.factory service.
   * @param \GuzzleHttp\ClientInterface $client
   *   The http_client service.
   * @param \Drupal\webform\Plugin\WebformElementManagerInterface $elementManager
   *   The plugin.manager.webform.element service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    RequestStack $requestStack,
    ConfigFactoryInterface $configFactory,
    LoggerChannelFactoryInterface $loggerFactory,
    ClientInterface $client,
    WebformElementManagerInterface $elementManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->requestStack = $requestStack;
    $this->configFactory = $configFactory;
    $this->loggerFactory = $loggerFactory;
    $this->client = $client;
    $this->elementManager = $elementManager;

    // This is very important for webform handler plugins, but is not documented
    // anywhere.
    $this->setConfiguration($configuration);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'url' => '',
      'elements' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getSummary() {
    $url = $this->getSetting('url');

    $settings[] = $this->t('Google form URL: %url', ['%url' => $url]);
    foreach ($this->getSetting('elements') as $info) {
      $settings[] = $this->t('%key is mapped to %value', [
        '%key' => $info['element_key'],
        '%value' => $info['entry_id'],
      ]);
    }

    return [
      '#theme' => 'item_list',
      '#items' => $settings,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['url'] = [
      '#type' => 'url',
      '#title' => $this->t('Google form URL'),
      '#size' => 30,
      '#default_value' => $this->getSetting('url'),
      '#element_validate' => [[self::class, 'validateUrlPattern']],
      '#required' => TRUE,
      '#weight' => 5,
    ];

    $form['elements'] = [
      '#type' => 'details',
      '#title' => $this->t('Map Google Form keys'),
      '#description' => $this->t('Provide the identifier for the entry in Google form i.e. entry.NUMBER. This can be done by inspecting the request when submitting a google form.'),
      '#open' => TRUE,
      '#tree' => TRUE,
      '#weight' => 6,
    ];

    $elements = $this->webform->getElementsInitializedAndFlattened();
    $settings = $this->getSetting('elements');

    foreach ($elements as $element_key => $element) {
      $element_plugin = $this->elementManager->getElementInstance($element);
      if (!$element_plugin->isInput($element) || !isset($element['#type'])) {
        continue;
      }
      if (!$element_plugin->isComposite()) {
        $index = $this->findArrayIndex($settings, $element_key);
        $form['elements'][$element_key] = [
          '#type' => 'textfield',
          '#title' => $this->t('Entry ID for @label', [
            '@label' => $element_plugin->getAdminLabel($element),
          ]),
          '#element_validate' => [[self::class, 'validateElementMap']],
          '#default_value' => is_int($index) ? $settings[$index]['entry_id'] : '',
        ];

        if ($element_plugin instanceof Email) {
          $form['elements'][$element_key]['#description'] = $this->t('If your Google form requires an email address, provide the value "emailAdress" here.');
          $form['elements'][$element_key]['#element_validate'][0] = [self::class, 'validateEmailElementMap'];
        }
      }
    }

    return $this->setSettingsParents($form);
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);

    // Prevent multiple emailAddress keys.
    $elements = array_filter($form_state->getValue('elements'), fn ($value) => $value === 'emailAddress');
    if (!empty($elements) && count($elements) > 1) {
      $form_state->setError($form, $this->t('You may only map one element to the "emailAddress" key.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    // This is help method is garbage and does not save any setting beacuse it
    // must first exist in configuration even though webform should be setting
    // default configuration from default configuration. WebformWTF.
    // $this->applyFormStateToConfiguration($form_state);
    $this->configuration['url'] = $form_state->getValue('url');
    $elements = array_filter($form_state->getValue('elements'), fn ($value) => strlen($value) > 0);
    $this->configuration['elements'] = array_map(fn ($key) => ['element_key' => $key, 'entry_id' => $elements[$key]], array_keys($elements));
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(WebformSubmissionInterface $webform_submission, $update = TRUE) {
    $url = $this->getSetting('url');
    $elements = $this->getSetting('elements');

    $values = [
      'submissionTimestamp' => (string) time(),
    ];
    $element_info = $this->webform->getElementsInitializedAndFlattened();
    $query = '';
    foreach ($elements as $info) {
      $element_key = $info['element_key'];
      $entry_id = $info['entry_id'];
      $element = $this->elementManager->getElementInstance($element_info[$element_key], $this->webform);
      $value = $webform_submission->getElementData($element_key);
      if ($element->hasMultipleValues($element_info) && is_array($value) && !empty($value)) {
        $values[$entry_id] = $value;
        foreach ($value as $index => $item_value) {
          if (!isset($element_info[$element_key]['#options'][$item_value])) {
            // If a multiple value is not in options, then the google form value
            // is "__other_option__" and an additional form key of
            // "entry.ID.other_option_response" is used to store the
            // user-provided value.
            $other_key = $entry_id . '.other_option_response';
            $values[$entry_id][$index] = '__other_option__';
            $values[$other_key] = $item_value;
          }
          else {
            $values[$entry_id][$index] = $item_value;
          }
        }

        if (count($values[$entry_id]) === 1) {
          // Reduce the multiple options if only one choice was selected.
          $values[$entry_id] = reset($values[$entry_id]);
        }
        else {
          // Multiple options values should be added as query parameters
          // instead.
          if (strlen($query) > 0) {
            $query .= '&';
          }
          $query .= Query::build([$entry_id => $values[$entry_id]]);
          unset($values[$entry_id]);
        }
      }
      else {
        $values[$entry_id] = $value;
      }
    }

    try {
      $request = $this->requestStack->getMainRequest();
      $headers = $request->headers->has('X-Forwarded-For')
        ? ['X-Forwarded-For' => $request->headers->get('X-Forwarded-For')]
        : [];
      $this->client->request('POST', $url, [
        'form_params' => $values,
        'headers' => $headers,
        'query' => $query,
      ]);

      if ($webform_submission->getWebform()->hasSubmissionLog()) {
        $context = [
          'link' => $webform_submission->id()
            ? $webform_submission->toLink($this->t('View'))->toString()
            : NULL,
          'webform_submission' => $webform_submission,
          'handler_id' => $this->getHandlerId(),
          'operation' => 'post to google form',
        ];
        $this
          ->getLogger('webform_submission')
          ->notice('Successfully posted to Google form', $context);
      }
      else {
        $context = [
          '@form' => $this->getWebform()->label(),
          'link' => $this->getWebform()->toLink(($this->t('Edit')))->toString(),
        ];
        $this
          ->getLogger('webform')
          ->notice('@form webform sent to google from', $context);
      }
    }
    catch (ClientException $e) {
      Error::logException($this->loggerFactory->get('webform_googleform'), $e);
    }
  }

  /**
   * Finds the array index for an arary.
   *
   * @param array $arr
   *   The associative array to search.
   * @param string $element_key
   *   The array key to match.
   *
   * @return int|null
   *   The array index or NULL if the key was not found.
   */
  protected function findArrayIndex(array $arr, string $element_key): ?int {
    foreach ($arr as $index => $info) {
      if ($info['element_key'] === $element_key) {
        return $index;
      }
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('request_stack'),
      $container->get('config.factory'),
      $container->get('logger.factory'),
      $container->get('http_client'),
      $container->get('plugin.manager.webform.element'),
    );
  }

  /**
   * Validates the element map value.
   *
   * @param array $element
   *   The form array element.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form state.
   */
  public static function validateElementMap(array $element, FormStateInterface $formState): void {
    if ($element['#value'] && !preg_match('/entry\.\d+/', $element['#value'])) {
      $formState->setError($element, new TranslatableMarkup('Invalid google form entry identifier.'));
    }
  }

  /**
   * Validates the element map value for email addresses.
   *
   * @param array $element
   *   The form array element.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form state.
   */
  public static function validateEmailElementMap(array $element, FormStateInterface $formState): void {
    if ($element['#value'] && $element['#value'] !== 'emailAddress' && !preg_match('/entry\.\d+/', $element['#value'])) {
      $formState->setError($element, new TranslatableMarkup('Invalid google form entry identifier.'));
    }
  }

  /**
   * Validates the google form URL pattern.
   *
   * @param array $element
   *   The form array element.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form state.
   */
  public static function validateUrlPattern(array $element, FormStateInterface $formState): void {
    $pattern = '/docs\.google\.com\/forms\/[a-zA-Z0-9_\-+\/]+\/formResponse$/';
    if ($element['#value'] && !preg_match($pattern, $element['#value'])) {
      $formState->setError($element, new TranslatableMarkup('Invalid google form URL'));
    }
  }

}
