<?php

namespace Drupal\Tests\webform_googlesheets\Functional\Handler;

use Drupal\Core\Database\Database;
use Drupal\google_api_client\Entity\GoogleApiServiceClient;
use Drupal\Tests\BrowserTestBase;
use Drupal\webform\Entity\Webform;

/**
 * Tests for Google Sheets webform handler functionality.
 *
 * @group webform
 */
class WebformGoogleSheetsTest extends BrowserTestBase {

  /**
   * Strict schema checking.
   *
   * Google API PHP Client module has an issue with schema.
   *
   * @var bool
   *
   * @see https://www.drupal.org/project/google_api_client/issues/3513218
   */
  protected $strictConfigSchema = FALSE;


  /**
   * Set default theme to stable.
   *
   * @var string
   */
  protected $defaultTheme = 'stable9';

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['webform', 'webform_googlesheets_test', 'dblog'];

  /**
   * Test the handler.
   */
  public function testGoogleSheetsHandler() {
    // If environmental variables aren't provided, skip the test.
    if (!$auth_config = base64_decode(getenv('GOOGLE_SERVICE_ACCOUNT_KEY'))) {
      $this->markTestSkipped('Unable to find or base64 decode environmental variable with Google API Service Account key.');
    }
    elseif (!$spreadsheet_url = base64_decode(getenv('GOOGLE_SHEETS_SPREADSHEET_URL'))) {
      $this->markTestSkipped('Unable to find environmental variable with Google Sheets Spreadsheet ID.');
    }

    // If pieces required for the test are missing, fail the test.
    if (!$service_account = GoogleApiServiceClient::load('webform_googlesheets_test')) {
      $this->fail('Google API Service Account entity not found.');
    }
    elseif (!$webform = Webform::load('test_googlesheets')) {
      $this->fail('Unable to load the test webform.');
    }
    elseif (!$handler = $webform->getHandler('google_sheets')) {
      $this->fail('Unable to find the handler for the test webform.');
    }

    // Set the Google Service account private key.
    $service_account->setAuthConfig($auth_config);
    $service_account->save();

    // Set spreadsheet.
    $handle_configuration = $handler->getConfiguration();
    if (preg_match('#https?://docs.google.com/spreadsheets/d/(?<id>.+?)/.+?(\#gid=(?<sheet_id>\d+))?$#', $spreadsheet_url, $matches)) {
      $handle_configuration['settings']['spreadsheet_url'] = $spreadsheet_url;
      $handle_configuration['settings']['spreadsheet_id'] = $matches['id'];
      $handle_configuration['settings']['spreadsheet_sheet_id'] = $matches['sheet_id'];
    }
    $handler->setConfiguration($handle_configuration);
    $webform->updateWebformHandler($handler);

    // Submit the Webform with Google Sheets handler.
    $this->drupalGet('/webform/' . $webform->id());
    $this->submitForm([], NULL);

    // Check if success message was added to the watchdog.
    $id = Database::getConnection()->select('watchdog', 'w')
      ->fields('w', ['wid'])
      ->condition('type', 'webform_submission')
      ->condition('message', 'Webform submission had been successfully appended to Google Sheets.')
      ->orderBy('wid', 'DESC')
      ->range(0, 1)
      ->execute()
      ->fetchField();
    $this->assertNotEquals(FALSE, $id, 'Log item was added confirming submission was appended to the Google Sheets.');
  }

}
