<?php

namespace Drupal\webform_headless\Controller;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\webform\WebformInterface;
use Drupal\webform_headless\Submitter;
use Drupal\webform_headless\WebformJsonSchemaManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for creating a submission.
 */
class CreateSubmissionController implements ContainerInjectionInterface {

  /**
   * The submitter.
   */
  protected readonly Submitter $submitter;

  /**
   * The schema manager.
   */
  protected readonly WebformJsonSchemaManager $schemaManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = new static();
    $instance->submitter = $container->get('webform_headless.submitter');
    $instance->schemaManager = $container->get('plugin.manager.webform_headless');

    return $instance;
  }

  /**
   * Creates a submission.
   */
  public function __invoke(WebformInterface $webform, Request $request): ?JsonResponse {
    $submission = $this->submitter->createSubmission($webform, $request);
    $schema = $this->schemaManager->getByRequest($request);

    return new JsonResponse([
      'uuid' => $submission->uuid(),
      'data' => $schema->denormalizeData($submission->getData(), $submission->getWebform()),
      'confirmation' => $this->submitter->getConfirmationSettings($submission),
    ]);
  }

}
