<?php

namespace Drupal\webform_headless\Controller;

use Drupal\Component\Utility\Crypt;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\File\Exception\FileException;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Lock\LockBackendInterface;
use Drupal\file\Upload\FileUploadHandler;
use Drupal\file\Upload\FormUploadedFile;
use Drupal\webform\WebformInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Lock\Exception\LockAcquiringException;
use Symfony\Component\Validator\ConstraintViolationInterface;

/**
 * Controller for uploading a file.
 */
class UploadFileController extends ControllerBase {

  /**
   * The file system service.
   */
  protected readonly FileSystemInterface $fileSystem;

  /**
   * The file upload handler.
   */
  protected readonly FileUploadHandler $fileUploadHandler;

  /**
   * The lock.
   */
  protected readonly LockBackendInterface $lock;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->fileSystem = $container->get('file_system');
    $instance->fileUploadHandler = $container->get('file.upload_handler');
    $instance->lock = $container->get('lock');

    return $instance;
  }

  /**
   * Uploads and saves an image from a Webform POST.
   *
   * @param \Drupal\webform\WebformInterface $webform
   *   The webform entity.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON object including the file URL.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\HttpException
   *   Thrown when file system errors occur.
   * @throws \Symfony\Component\HttpKernel\Exception\UnprocessableEntityHttpException
   *   Thrown when validation errors occur.
   */
  public function __invoke(WebformInterface $webform, Request $request): Response {
    // Check if multipart/form-data.
    if ($request->getContentTypeFormat() !== 'form') {
      return new JsonResponse(
        ['errors' => [['message' => 'Content type must be application/x-www-form-urlencoded or multipart/form-data']]],
        Response::HTTP_BAD_REQUEST
      );
    }

    $form = $webform->getSubmissionForm();
    $results = [];

    // Getting the UploadedFile directly from the request.
    foreach ($request->files->get('files') as $name => $uploads) {
      foreach ($uploads as $upload) {
        assert($upload instanceof UploadedFile);

        $element = $webform->getElement($name);
        if ($element === NULL) {
          return new JsonResponse(
            ['errors' => [['message' => sprintf("No element with name '%s' exists.", $name)]]],
            Response::HTTP_BAD_REQUEST,
          );
        }

        $builtElement = NestedArray::getValue($form, ['elements', ...$element['#webform_parents']]);
        $destination = $builtElement['#upload_location'];

        // Check the destination file path is writable.
        if (!$this->fileSystem->prepareDirectory($destination, FileSystemInterface::CREATE_DIRECTORY)) {
          return new JsonResponse(
            ['errors' => [['message' => 'Destination file path is not writable']]],
            Response::HTTP_INTERNAL_SERVER_ERROR
          );
        }

        $filename = $upload->getClientOriginalName();
        $fileUri = "{$destination}/{$filename}";
        $fileUri = $this->fileSystem->getDestinationFilename($fileUri, FileExists::Rename);

        // Lock based on the prepared file URI.
        $lock_id = $this->generateLockIdFromFileUri($fileUri);
        if (!$this->lock->acquire($lock_id)) {
          return new JsonResponse(
            ['errors' => [['message' => sprintf('File "%s" is already locked for writing.', $fileUri)]]],
            Response::HTTP_SERVICE_UNAVAILABLE,
            ['Retry-After' => 1],
          );
        }

        // Validate and handle the uploaded file.
        $validators = $builtElement['#upload_validators'] ?? [];

        try {
          $uploadedFile = new FormUploadedFile($upload);
          $uploadResult = $this->fileUploadHandler->handleFileUpload($uploadedFile, $validators, $destination, FileExists::Rename, FALSE);
          if ($uploadResult->hasViolations()) {
            return new JsonResponse(
              [
                'errors' => array_map(
                  fn (ConstraintViolationInterface $violation) => ['message' => $violation->getMessage()],
                  iterator_to_array($uploadResult->getViolations()),
                ),
              ],
              Response::HTTP_BAD_REQUEST,
            );
          }
        }
        catch (FileException $e) {
          return new JsonResponse(
            ['errors' => [['message' => 'File could not be saved']]],
            Response::HTTP_INTERNAL_SERVER_ERROR,
          );
        }
        catch (LockAcquiringException $e) {
          return new JsonResponse(
            ['errors' => [['message' => sprintf('File "%s" is already locked for writing.', $upload->getClientOriginalName())]]],
            Response::HTTP_SERVICE_UNAVAILABLE,
            ['Retry-After' => 1],
          );
        }

        $this->lock->release($lock_id);
        $file = $uploadResult->getFile();
        $results[$name][] = $file->id();
      }
    }

    return new JsonResponse($results, Response::HTTP_CREATED);
  }

  /**
   * Generates a lock ID based on the file URI.
   *
   * @param string $file_uri
   *   The file URI.
   *
   * @return string
   *   The generated lock ID.
   */
  protected static function generateLockIdFromFileUri(string $file_uri): string {
    return 'file:webform_headless:' . Crypt::hashBase64($file_uri);
  }

}
