<?php

namespace Drupal\webform_headless\Plugin\WebformJsonSchema;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Datetime\Entity\DateFormat;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\webform\Plugin\WebformElement\BooleanBase;
use Drupal\webform\Plugin\WebformElement\WebformManagedFileBase;
use Drupal\webform\Utility\WebformDateHelper;
use Drupal\webform\Utility\WebformElementHelper;
use Drupal\webform\WebformInterface;
use Drupal\webform\WebformSubmissionConditionsValidator;
use Drupal\webform_headless\Attribute\WebformJsonSchema;
use Drupal\webform_headless\WebformItem;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a plugin for FormKit.
 *
 * @see https://formkit.com
 */
#[WebformJsonSchema(
  id: "form_kit",
  label: new TranslatableMarkup("FormKit"),
)]
class FormKit extends WebformJsonSchemaBase {

  protected const FILE_ELEMENTS = [
    'managed_file',
    'webform_audio_file',
    'webform_document_file',
    'webform_image_file',
    'webform_video_file',
  ];

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The language manager.
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->moduleHandler = $container->get('module_handler');
    $instance->languageManager = $container->get('language_manager');
    $instance->configFactory = $container->get('config.factory');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function normalizeData(array $data, WebformInterface $webform): array {
    $data = parent::normalizeData($data, $webform);

    foreach ($data as $key => &$value) {
      $item = $this->webformItemTransformer->getWebformItemFromWebform($webform, $key);
      if ($item === NULL) {
        continue;
      }

      // Normalize non-composite, multiple element values.
      if (!empty($item->element['#multiple']) && $item->children === [] && !$item->elementPlugin instanceof WebformManagedFileBase) {
        foreach ($value as &$singleValue) {
          if (is_array($singleValue)) {
            $singleValue = reset($singleValue);
          }
        }
      }

      $dateFormat = $item->getElementProperty('date_date_format')
        ?? DateFormat::load('html_date')?->getPattern();
      $timeFormat = $item->getElementProperty('date_time_format')
        ?? DateFormat::load('html_time')?->getPattern();

      if ($item->element['#type'] === 'datetime') {
        $dateTime = WebformDateHelper::createFromFormat($dateFormat, $value);
        $value = [
          'date' => $dateTime?->format($dateFormat),
          'time' => $dateTime?->format($timeFormat),
        ];
      }

      if ($item->element['#type'] === 'date') {
        $dateTime = WebformDateHelper::createFromFormat($dateFormat, $value) ?: NULL;
        $value = $dateTime?->format($dateFormat);
      }

      if ($item->element['#type'] === 'webform_time') {
        $dateTime = DrupalDateTime::createFromTimestamp(strtotime($value)) ?: NULL;
        $value = $dateTime?->format($timeFormat);
      }

      if ($item->elementPlugin instanceof BooleanBase) {
        $value = (int) filter_var($value, FILTER_VALIDATE_BOOLEAN);
      }
    }

    return $data;
  }

  /**
   * {@inheritdoc}
   */
  protected function buildSchema(WebformInterface $webform, array $items): array {
    $elements = $this->getElementsFromWebformItems($webform, $items);

    $schema = [
      '$cmp' => 'FormKit',
      'props' => [
        'type' => 'form',
        'id' => $webform->id(),
      ],
      'children' => $elements,
    ];

    // Check if the form has a multi-step element.
    foreach ($schema['children'] as $key => $child) {
      if (!isset($child['$formkit'])) {
        continue;
      }

      if ($child['$formkit'] === 'multi-step') {
        $multiStepKey = $key;
      }

      if ($child['$formkit'] === 'submit') {
        $submitElementKey = $key;
      }
    }

    if (isset($multiStepKey)) {
      $submitLabel = $this->t('Submit');

      // Remove any actions element from the schema.
      if (isset($submitElementKey)) {
        if (isset($schema['children'][$submitElementKey]['label'])) {
          $submitLabel = $schema['children'][$submitElementKey]['label'];
        }
        unset($schema['children'][$submitElementKey]);
      }

      // Add the submit button to the last step.
      $lastStepKey = count($schema['children'][$multiStepKey]['children']) - 1;
      $schema['children'][$multiStepKey]['children'][$lastStepKey]['__raw__sectionsSchema'] = [
        'stepNext' => [
          // @see https://github.com/formkit/formkit/issues/1240#issuecomment-1981049714
          'if' => 'true',
          'children' => [
            [
              '$formkit' => 'submit',
              'label' => $submitLabel,
            ],
          ],
        ],
      ];
    }

    // Disable the default submit button.
    if ($webform->hasActions() || isset($multiStepKey)) {
      $schema['props']['actions'] = FALSE;
    }

    $this->moduleHandler->alter('webform_headless_formkit_schema', $webform, $schema);

    return [$schema];
  }

  /**
   * Gets the elements from webform items.
   *
   * @param \Drupal\webform\WebformInterface $webform
   *   The webform.
   * @param \Drupal\webform_headless\WebformItem[] $items
   *   The webform items.
   * @param string[] $parents
   *   The parents.
   */
  protected function getElementsFromWebformItems(WebformInterface $webform, array $items, array $parents = []): array {
    $elements = [];

    foreach ($items as $key => $item) {

      // Check element access and skip elements that are not accessible.
      if (isset($item->element['#access']) && $item->element['#access'] === FALSE) {
        continue;
      }

      $properties = [];
      $properties['name'] = $key;

      // The id prop is needed to be able to use the $get function to access
      // the context object of FormKit Inputs.
      // @see https://formkit.com/essentials/schema#accessing-other-inputs
      $properties['id'] = $item->element['#webform_id']
        ?? $item->element['#webform_composite_id'];

      // Conditional or iterative (when using if or for) schema nodes should
      // always include an explicit key prop.
      // @see https://formkit.com/essentials/schema#conditionals
      $properties['key'] = $item->element['#webform_id']
        ?? $item->element['#webform_composite_id'];

      if (isset($item->element['#title'])) {
        $properties['label'] = $item->element['#title'];
      }

      if (isset($item->builtElement['#title_display'])) {
        if (in_array($item->builtElement['#title_display'], ['after', 'inline', 'invisible'])) {
          $properties['wrapperClass'] = 'label-display-' . $item->builtElement['#title_display'];
        }
        if ($item->builtElement['#title_display'] === 'none') {
          unset($properties['label']);
        }
      }

      if (!empty($item->builtElement['#description'])) {
        $properties['help'] = strip_tags(WebformElementHelper::convertToString($item->builtElement['#description']));
      }

      if (isset($item->builtElement['#description_display'])) {
        if ($item->builtElement['#description_display'] === 'invisible') {
          $properties['helpClass'] = 'help-display-invisible';
        }
        if ($item->builtElement['#description_display'] === 'before') {
          $properties['outerClass'] = 'help-display-before';
          $properties['__raw__sectionsSchema']['wrapper'] = [
            '$el' => null,
          ];
        }
        // TODO: Support description_display = tooltip.
      }

      if (isset($item->builtElement['#placeholder'])) {
        $properties['placeholder'] = $item->builtElement['#placeholder'];
      }

      if (isset($item->builtElement['#attributes']['class'])) {
        $properties['attrs']['class'] = implode(' ', $item->builtElement['#attributes']['class']);
      }

      if (isset($item->builtElement['#field_prefix'])) {
        $properties['prefix'] = $item->builtElement['#field_prefix'];
      }

      if (isset($item->builtElement['#field_suffix'])) {
        $properties['suffix'] = $item->builtElement['#field_suffix'];
      }

      if (isset($item->element['#type'])) {
        $booleanElements = [
          'checkbox',
          'webform_terms_of_service',
        ];

        if (in_array($item->element['#type'], $booleanElements)) {
          $properties['$formkit'] = 'checkbox';

          // A required boolean value in Webform means 'value is TRUE'.
          // In FormKit, it means 'value is passed and is TRUE or FALSE'.
          if (isset($item->element['#required']) && $item->element['#required'] === TRUE) {
            $properties['validation'][] = ['is', TRUE];
          }
        }

        if ($item->element['#type'] === 'textarea') {
          $properties['$formkit'] = 'textarea';

          $this->addTextMinMaxLengthProperties($item, $properties);
          $this->addRegexProperties($item, $properties);
        }

        if ($item->element['#type'] === 'textfield') {
          $properties['$formkit'] = 'text';

          $this->addTextMinMaxLengthProperties($item, $properties);
          $this->addRegexProperties($item, $properties);
        }

        if ($item->element['#type'] === 'hidden') {
          $properties['$formkit'] = 'hidden';
          $this->addTextMinMaxLengthProperties($item, $properties);
          $this->addRegexProperties($item, $properties);
        }

        if ($item->element['#type'] === 'webform_horizontal_rule') {
          $properties['$el'] = 'hr';
        }

        if ($item->element['#type'] === 'webform_markup') {
          $properties['$el'] = 'div';
          $properties['attrs']['innerHTML'] = $item->element['#markup'];
        }

        if ($item->element['#type'] === 'select') {
          $properties['$formkit'] = 'select';
          $properties['options'] = array_map(
            fn ($key, $value) => ['label' => $value, 'value' => $key],
            array_keys($item->element['#options']),
            array_values($item->element['#options']),
          );
          if (isset($item->element['#empty_option'])) {
            $properties['placeholder'] = $item->element['#empty_option'];
          }
        }

        if ($item->element['#type'] === 'checkboxes') {
          $properties['$formkit'] = 'checkbox';
          $properties['options'] = array_map(
            fn ($key) => [
              'label' => $item->builtElement[$key]['#title'],
              'value' => (string) $item->builtElement[$key]['#return_value'],
              'help' => $item->builtElement[$key]['#description'] ?? NULL,
            ],
            Element::children($item->builtElement),
          );
        }

        if ($item->element['#type'] === 'radios') {
          $properties['$formkit'] = 'radio';
          $properties['options'] = array_map(
            fn ($key) => [
              'label' => $item->builtElement[$key]['#title'],
              'value' => (string) $item->builtElement[$key]['#return_value'],
              'help' => $item->builtElement[$key]['#description'] ?? NULL,
            ],
            Element::children($item->builtElement),
          );
        }

        if ($item->element['#type'] === 'number') {
          $properties['$formkit'] = 'number';
          $this->addNumberMinMaxStepProperties($item, $properties);
        }

        if ($item->element['#type'] === 'range') {
          $properties['$formkit'] = 'range';
          $this->addNumberMinMaxStepProperties($item, $properties);
        }

        if ($item->element['#type'] === 'email') {
          $properties['$formkit'] = 'email';
          $properties['validation'][] = ['email'];
          $this->addTextMinMaxLengthProperties($item, $properties);
          $this->addRegexProperties($item, $properties);
        }

        if ($item->element['#type'] === 'url') {
          $properties['$formkit'] = 'url';
          $properties['validation'][] = ['url'];
          $this->addTextMinMaxLengthProperties($item, $properties);
          $this->addRegexProperties($item, $properties);
        }

        if ($item->element['#type'] === 'tel') {
          $properties['$formkit'] = 'tel';
          $this->addTextMinMaxLengthProperties($item, $properties);
          $this->addRegexProperties($item, $properties);
        }

        if ($item->element['#type'] === 'datetime') {
          $properties['$formkit'] = 'date';
          $this->addDateMinMaxProperties($item, $properties);

          $dateElement = $item->getElementProperty('date_date_element');
          if (in_array($dateElement, ['datetime', 'datetime-local'])) {
            // FormKit doesn't support the non-localized HTML5 datetime.
            $properties['$formkit'] = 'datetime-local';
          }
          if ($dateElement === 'datepicker') {
            // Requires the webform_jqueryui_datepicker module.
            $properties['$formkit'] = 'datepicker';
            $this->addDatePickerProperties($item, $properties);
          }

          // @todo Support 'None' / 'Text input' elements
        }

        if ($item->element['#type'] === 'date') {
          $properties['$formkit'] = 'date';
          $this->addDateMinMaxProperties($item, $properties);

          if ($item->getElementProperty('datepicker')) {
            // Requires the webform_jqueryui_datepicker module.
            $properties['$formkit'] = 'datepicker';
            $this->addDatePickerProperties($item, $properties);
          }
        }

        if ($item->element['#type'] === 'webform_time') {
          $properties['$formkit'] = 'time';
          $this->addTimeMinMaxStepProperties($item, $properties);
        }

        if (in_array($item->element['#type'], static::FILE_ELEMENTS)) {
          $extensions = $item->elementPlugin->getDefaultProperty('file_extensions');
          $extensions = explode(' ', $extensions);
          $extensions = '.' . implode(',.', $extensions);

          $properties['$formkit'] = 'file';
          $properties['accept'] = $extensions;
          $properties['multiple'] = !empty($item->element['#multiple']);

          if ($item->element['#type'] === 'webform_audio_file') {
            $properties['file-item-icon'] = $properties['no-files-icon'] = 'fileAudio';
          }

          if ($item->element['#type'] === 'webform_document_file') {
            $properties['file-item-icon'] = $properties['no-files-icon'] = 'fileDoc';
          }

          if ($item->element['#type'] === 'webform_image_file') {
            $properties['file-item-icon'] = $properties['no-files-icon'] = 'fileImage';
          }

          if ($item->element['#type'] === 'webform_video_file') {
            $properties['file-item-icon'] = $properties['no-files-icon'] = 'fileVideo';
          }
        }

        if ($item->element['#type'] === 'webform_belgian_national_insurance_number') {
          $properties['$formkit'] = 'mask';
          $properties['mask'] = '##.##.##-###.##';

          $regex = '^[0-9]{2}[-\\.]{0,1}[0-9]{2}[-\\.]{0,1}[0-9]{2}[-\\.]{0,1}[0-9]{3}[-\\.]{0,1}[0-9]{2}$';
          $properties['validation'][] = ['matches', "/$regex/"];

          if (isset($item->builtElement['#error_message'])) {
            $properties['validation-messages']["matches:/$regex/"] = $item->builtElement['#error_message'];
          }
        }

        // @todo Support min/max.
        // @todo Support days of the week validation.
        if ($item->element['#type'] === 'webform_actions') {
          $properties['$formkit'] = 'submit';
          if (isset($item->element['#submit__label'])) {
            $properties['label'] = $item->element['#submit__label'];
          }
          // Use the form's disabled status.
          $properties['bind'] = '$submitAttrs';
          $properties['disabled'] = '$disabled';
        }
      }

      // We need to check the pre-built element because of \Drupal\webform\Plugin\WebformElement\Checkbox::hiddenElementAfterBuild().
      if (isset($item->element['#required']) && $item->element['#required'] === TRUE) {
        $properties['validation'][] = ['required'];
        if (isset($item->element['#required_error'])) {
          $properties['validation-messages']['required'] = $item->element['#required_error'];
        }
      }

      if (isset($item->element['#disabled']) && $item->element['#disabled'] === TRUE) {
        $properties['disabled'] = TRUE;
      }

      if (!empty($item->element['#states'])) {
        $this->applyStates($item->element['#states'], $properties, $webform);
      }

      if ($item->children !== []) {
        $children = $this->getElementsFromWebformItems(
          $webform,
          $item->children,
          [...$parents, $key],
        );

        if ($item->element['#type'] === 'webform_wizard_page') {
          $properties['$formkit'] = 'step';
          $properties['allow-incomplete'] = FALSE;

          if (isset($item->element['#prev_button_label'])) {
            $properties['previous-label'] = $item->element['#prev_button_label'];
          }

          if (isset($item->element['#next_button_label'])) {
            $properties['next-label'] = $item->element['#next_button_label'];
          }

          $properties['children'] = $children;
        }
        elseif ($item->element['#type'] === 'webform_flexbox') {
          $properties = [
            '$el' => 'div',
            'attrs' => [
              'class' => 'webform-flexbox webform-flexbox--' . ($item->element['#align_items'] ?? 'flex-start'),
            ],
            'children' => $children,
          ];
        }
        else {
          $properties['$formkit'] = 'group';
          $properties['children'] = $children;
        }
      }

      if (!empty($item->element['#multiple']) && !in_array($item->element['#type'], static::FILE_ELEMENTS)) {
        $repeaterProperties = $properties;
        $repeaterProperties['$formkit'] = 'repeater';
        $repeaterProperties['max'] = $item->element['#multiple'];

        if (isset($item->element['#multiple__min_items'])) {
          $repeaterProperties['min'] = $item->element['#multiple__min_items'];
        }

        if (isset($item->element['#multiple__sorting'])) {
          $repeaterProperties['draggable'] = (bool) $item->element['#multiple__sorting'];
        }

        if (isset($item->element['#multiple__add_more'])) {
          $repeaterProperties['add-button'] = (bool) $item->element['#multiple__add_more'];
        }

        if (isset($item->builtElement['#add_more_button_label'])) {
          $repeaterProperties['add-label'] = $item->builtElement['#add_more_button_label'];
        }

        if (isset($item->builtElement['#multiple__operations'])) {
          $repeaterProperties['insert-control'] = (bool) $item->builtElement['#multiple__operations'];
          $repeaterProperties['remove-control'] = (bool) $item->builtElement['#multiple__operations'];

          if (isset($item->builtElement['#multiple__add'])) {
            $repeaterProperties['insert-control'] = (bool) $item->builtElement['#multiple__add'];
          }

          if (isset($item->builtElement['#multiple__remove'])) {
            $repeaterProperties['remove-control'] = (bool) $item->builtElement['#multiple__remove'];
          }
        }

        if (empty($repeaterProperties['children'])) {
          $repeaterProperties['children'] = [
            ['name' => "{$properties['name']}[value]"] + $properties,
          ];
        }

        $properties = $repeaterProperties;
      }

      $this->moduleHandler->alter('webform_headless_formkit_webform_item_schema', $item, $properties);

      if (!isset($properties['$formkit']) && !isset($properties['$cmp']) && !isset($properties['$el'])) {
        continue;
      }

      $elements[] = $properties;
    }

    // Wrap subsequent steps in a multi-step component.
    foreach ($elements as $i => &$element) {
      if (isset($elements[$i - 1])) {
        $previousElement = &$elements[$i - 1];
      }
      else {
        unset($previousElement);
        $previousElement = NULL;
      }

      if (!isset($element['$formkit']) || $element['$formkit'] !== 'step') {
        continue;
      }

      if (!$previousElement || $previousElement['$formkit'] !== 'multi-step') {
        // Wrap the first step in a multi-step component.
        $element = [
          '$formkit' => 'multi-step',
          'name' => 'multi_step',
          'id' => 'multi_step',
          'tab-style' => $webform->getSetting('wizard_progress_bar') ? 'progress' : 'none',
          'children' => [$element],
        ];
      }
      else {
        // Add the step to the multi-step component.
        $previousElement['children'][] = $element;
        unset($elements[$i]);
        $elements = array_values($elements);
      }
    }

    return $elements;
  }

  /**
   * Gets the min, max, and step properties for number-like elements.
   */
  protected function addNumberMinMaxStepProperties(WebformItem $item, array &$properties): void {
    $min = $item->getElementProperty('min');
    if ($min !== NULL && $min !== '') {
      $properties['min'] = $min;
      $properties['validation'][] = ['min', $min];
    }

    $max = $item->getElementProperty('max');
    if ($max !== NULL && $max !== '') {
      $properties['max'] = $max;
      $properties['validation'][] = ['max', $max];
    }

    $step = $item->getElementProperty('step');
    if ($step !== NULL && $step !== '') {
      $properties['step'] = $step;
    }
    else {
      $properties['step'] = 'any';
    }
  }

   /**
   * Gets regex and validation message.
   */
  protected function addRegexProperties(WebformItem $item, array &$properties): void {
    if (isset($item->element['#pattern'])) {
      $properties['validation'][] = ['matches', '/' . $item->element['#pattern'] . '/'];

      if (isset($item->element['#pattern_error'])) {
        $properties['validation-messages']['matches'] = $item->element['#pattern_error'];
      }
    }
  }

   /**
   * Gets the min, max length for text-like elements.
   */
  protected function addTextMinMaxLengthProperties(WebformItem $item, array &$properties): void {
    $min = $item->getElementProperty('minlength');
    $max = $item->getElementProperty('maxlength');

    if ($max !== NULL && $min !== NULL) {
      $properties['minlength'] = $min;
      $properties['maxlength'] = $max;
      $properties['validation'][] = ['length', $min, $max];
    }
    else if ($min !== NULL) {
      $properties['minlength'] = $min;
      $properties['validation'][] = ['length', $min];
    }
    else if ($max !== NULL) {
      $properties['maxlength'] = $max;
      $properties['validation'][] = ['length', 0, $max];
    }
  }

  /**
   * Gets the min and max properties for date-like elements.
   */
  protected function addDateMinMaxProperties(WebformItem $item, array &$properties): void {
    if ($minDate = $item->getElementProperty('date_date_min')) {
      if ($timestamp = strtotime($minDate)) {
        $dateTime = \DateTime::createFromFormat('U', $timestamp);
        $properties['min'] = $dateTime->format('Y-m-d');
      }
    }

    if ($maxDate = $item->getElementProperty('date_date_max')) {
      if ($timestamp = strtotime($maxDate)) {
        $dateTime = \DateTime::createFromFormat('U', $timestamp);
        $properties['max'] = $dateTime->format('Y-m-d');
      }
    }
  }

  /**
   * Gets the min, max and step properties for time elements.
   */
  protected function addTimeMinMaxStepProperties(WebformItem $item, array &$properties): void {
    $min = $item->getElementProperty('min');
    if ($min !== NULL && $min !== '') {
      $properties['min'] = $min;
    }

    $max = $item->getElementProperty('max');
    if ($max !== NULL && $max !== '') {
      $properties['max'] = $max;
    }

    $step = $item->getElementProperty('step');
    if ($step !== NULL && $step !== '') {
      $properties['step'] = $step;
    }
    else {
      $properties['step'] = 'any';
    }
  }

  /**
   * Gets date picker related properties for date-like elements.
   */
  protected function addDatePickerProperties(WebformItem $item, array &$properties): void {
    $dateFormat = $item->getElementProperty('date_date_format');
    $properties['format'] = $properties['value-format'] = $this->convertDateFormat($dateFormat);
    $properties['value-locale'] = $this->languageManager->getCurrentLanguage()->getId();

    // Set first day according to admin/config/regional/settings.
    $config = $this->configFactory->get('system.date');
    $properties['week-start'] = (string) $config->get('first_day');

    if (isset($properties['min'])) {
      $properties['min-date'] = $properties['min'];
      unset($properties['min']);
    }

    if (isset($properties['max'])) {
      $properties['max-date'] = $properties['max'];
      unset($properties['max']);
    }
  }

  /**
   * Converts a PHP date format to a JS date format.
   */
  protected function convertDateFormat(string $format): string {
    $tokenMap = [
      'd' => 'DD',
      'D' => 'ddd',
      'j' => 'D',
      'l' => 'dddd',
      'N' => 'E',
      'S' => 'o',
      'w' => 'e',
      'z' => 'DDD',
      'W' => 'W',
      'F' => 'MMMM',
      'm' => 'MM',
      'M' => 'MMM',
      'n' => 'M',
      't' => 'D',
      'L' => 'L',
      'o' => 'YYYY',
      'Y' => 'YYYY',
      'y' => 'YY',
      'a' => 'a',
      'A' => 'A',
      'B' => 'SSS',
      'g' => 'h',
      'G' => 'H',
      'h' => 'hh',
      'H' => 'HH',
      'i' => 'mm',
      's' => 'ss',
      'u' => 'SSS',
      'e' => 'zz',
      'I' => 'ZZ',
      'O' => 'ZZ',
      'P' => 'Z',
      'T' => 'z',
      'Z' => 'X',
      'c' => 'YYYY-MM-DDTHH:mm:ssZ',
      'r' => 'ddd, DD MMM YYYY HH:mm:ss ZZ',
      'U' => 'X',
    ];

    $tokens = str_split($format);
    $tokens = array_map(fn ($token) => $tokenMap[$token] ?? $token, $tokens);

    return implode('', $tokens);
  }

  /**
   * Converts Drupal form #states to conditions and applies them to properties.
   *
   * @param array $states
   *   The states.
   * @param array $properties
   *   The properties.
   * @param \Drupal\webform\WebformInterface $webform
   *   The webform.
   */
  protected function applyStates(array $states, array &$properties, WebformInterface $webform): void {
    foreach ($states as $state => $triggersBySelector) {
      if ($state === 'visible') {
        $properties['if'] = $this->getCondition($triggersBySelector, FALSE, $webform);
      }
      elseif ($state === 'invisible') {
        $properties['if'] = $this->getCondition($triggersBySelector, TRUE, $webform);
      }
      elseif ($state === 'enabled') {
        $properties['disabled'] = $this->getCondition($triggersBySelector, TRUE, $webform);
      }
      elseif ($state === 'disabled') {
        $properties['disabled'] = $this->getCondition($triggersBySelector, FALSE, $webform);
      }
      elseif ($state === 'required' || $state === 'optional') {
        if (!isset($properties['validation'])) {
          $properties['validation'] = [];
        }
        if (!in_array(['required'], $properties['validation'], FALSE)) {
          $properties['validation'] = [
            'if' => $this->getCondition($triggersBySelector, $state === 'optional', $webform),
            'then' => [...$properties['validation'], ['required']],
            'else' => $properties['validation'],
          ];
        }
      }
      else {
        throw new \LogicException(sprintf('The state "%s" is not supported.', $state));
      }

      // @todo Implement the 'visible-slide' state.
      // @todo Implement the 'invisible-slide' state.
      // @todo Implement the 'readwrite' state.
      // @todo Implement the 'readonly' state.
      // @todo Implement the 'expanded' state.
      // @todo Implement the 'collapsed' state.
      // @todo Implement the 'checked' state.
      // @todo Implement the 'unchecked' state.
    }
  }

  /**
   * Converts Drupal form #states to a condition.
   *
   * @param array $triggersBySelector
   *   The #states triggers, grouped by selector.
   * @param bool $inverted
   *   Whether the condition should be inverted.
   * @param \Drupal\webform\WebformInterface $webform
   *   The webform.
   *
   * @return string
   *   The condition.
   */
  protected function getCondition(array $triggersBySelector, bool $inverted, WebformInterface $webform): string {
    $condition = '';

    foreach ($triggersBySelector as $selector => $triggers) {
      if ($triggers === 'or') {
        if ($inverted) {
          $condition .= ' && ';
        }
        else {
          $condition .= ' || ';
        }
        continue;
      }

      if ($triggers === 'and') {
        if ($inverted) {
          $condition .= ' || ';
        }
        else {
          $condition .= ' && ';
        }
        continue;
      }

      if ($triggers === 'xor') {
        // @todo Implement the 'xor' conjunction.
        throw new \LogicException('The "xor" conjunction is not supported.');
      }

      $name = WebformSubmissionConditionsValidator::getSelectorInputName($selector);

      if ($name === NULL) {
        $condition .= $this->getCondition($triggers, $inverted, $webform);
        continue;
      }

      $arrayValue = WebformSubmissionConditionsValidator::getInputNameAsArray($name, 1);
      if ($arrayValue !== NULL) {
        $name = WebformSubmissionConditionsValidator::getInputNameAsArray($name, 0);
      }

      $element = $webform->getElement($name);
      if ($element === NULL) {
        throw new \LogicException(sprintf('The element "%s" does not exist.', $name));
      }
      if (!isset($element['#webform_id'])) {
        throw new \LogicException(sprintf('The element "%s" does not have a #webform_id.', $name));
      }
      $id = $element['#webform_id'];

      foreach ($triggers as $trigger => $value) {
        if ($trigger === 'empty') {
          $operator = '==';
          $endValue = '""';
        }
        elseif ($trigger === 'filled') {
          $operator = NULL;
          $endValue = NULL;
        }
        elseif ($trigger === 'checked') {
          $operator = '==';
          if ($arrayValue !== NULL) {
            $endValue = sprintf("'%s'", $arrayValue);
          }
          else {
            $endValue = 'true';
          }
        }
        elseif ($trigger === 'unchecked') {
          if ($arrayValue !== NULL) {
            $operator = '!=';
            $endValue = sprintf("'%s'", $arrayValue);
          }
          else {
            $operator = '==';
            $endValue = 'false';
          }
        }
        elseif ($trigger === 'value' && is_string($value)) {
          $operator = '===';
          $endValue = sprintf("'%s'", $value);
        }
        elseif ($trigger === 'value' && is_array($value)) {
          $condition .= $this->getCondition([$selector => $value], $inverted, $webform);
          continue;
        }
        elseif ($trigger === '!value') {
          $operator = '!==';
          $endValue = sprintf("'%s'", $value);
        }
        elseif ($trigger === 'less') {
          $operator = '<';
          $endValue = sprintf("'%s'", $value);
        }
        elseif ($trigger === 'less_equal') {
          $operator = '<=';
          $endValue = sprintf("'%s'", $value);
        }
        elseif ($trigger === 'greater') {
          $operator = '>';
          $endValue = sprintf("'%s'", $value);
        }
        elseif ($trigger === 'greater_equal') {
          $operator = '>=';
          $endValue = sprintf("'%s'", $value);
        }
        elseif ($trigger === 'between') {
          [$start, $end] = explode(':', $value);
          $this->appendCondition($condition, $id, '>=', $start, $inverted);
          $this->appendCondition($condition, $id, '<=', $end, $inverted);
          continue;
        }
        elseif ($trigger === '!between') {
          [$start, $end] = explode(':', $value);
          $this->appendCondition($condition, $id, '>=', $start, !$inverted);
          $this->appendCondition($condition, $id, '<=', $end, !$inverted);
          continue;
        }
        else {
          throw new \LogicException(sprintf('The trigger "%s" is not supported.', $trigger));
        }

        if (!empty($element['#webform_multiple'])) {
          $operator = match ($operator) {
            '==' => 'CONTAINS',
            '===' => 'CONTAINS_STRICT',
            '!=' => '!CONTAINS',
            '!==' => '!CONTAINS_STRICT',
            default => $operator,
          };
        }

        // @todo Implement the 'pattern' trigger.
        // @todo Implement the '!pattern' trigger.
        $this->appendCondition($condition, $id, $operator, $endValue, $inverted);
      }
    }

    return $condition;
  }

  /**
   * Joins another condition to a condition string.
   *
   * @param string $condition
   *   The condition string.
   * @param string $id
   *   The ID of the element to compare.
   * @param string|null $operator
   *   The operator to use.
   * @param string|null $value
   *   The value to compare against.
   * @param bool $inverted
   *   Whether the condition should be inverted.
   */
  protected function appendCondition(string &$condition, string $id, ?string $operator, ?string $value, bool $inverted = FALSE): void {
    if ($inverted) {
      $operator = match ($operator) {
        '==' => '!=',
        '!=' => '==',
        '===' => '!==',
        '!==' => '===',
        '<' => '>=',
        '<=' => '>',
        '>' => '<=',
        '>=' => '<',
        default => $operator,
      };
    }

    if ($condition !== '' && (!str_ends_with($condition, ' || ') || !str_ends_with($condition, ' && '))) {
      if ($inverted) {
        $condition .= ' || ';
      }
      else {
        $condition .= ' && ';
      }
    }

    if ($operator === 'CONTAINS') {
      $condition .= sprintf("\$helpers.contains(\$get('%s').value, %s, false)", $id, $value);
    }
    elseif ($operator === 'CONTAINS_STRICT') {
      $condition .= sprintf("\$helpers.contains(\$get('%s').value, %s, true)", $id, $value);
    }
    elseif ($operator === '!CONTAINS') {
      $condition .= sprintf("!\$helpers.contains(\$get('%s').value, %s, false)", $id, $value);
    }
    elseif ($operator === '!CONTAINS_STRICT') {
      $condition .= sprintf("!\$helpers.contains(\$get('%s').value, %s, true)", $id, $value);
    }
    else {
      $condition .= sprintf("\$get('%s').value", $id);
      if ($operator && $value !== null) {
        $condition .= sprintf(' %s %s', $operator, $value);
      }
    }
  }

}
