<?php

namespace Drupal\webform_headless\Plugin\WebformJsonSchema;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\file\FileInterface;
use Drupal\webform\Plugin\WebformElement\BooleanBase;
use Drupal\webform\Plugin\WebformElement\WebformCompositeBase;
use Drupal\webform\Plugin\WebformElement\WebformImageFile;
use Drupal\webform\WebformInterface;
use Drupal\webform\WebformSubmissionInterface;
use Drupal\webform_headless\Transformer\WebformItemTransformer;
use Drupal\webform_headless\WebformItem;
use Drupal\webform_headless\WebformJsonSchemaInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Base class for WebformJsonSchema plugins.
 */
abstract class WebformJsonSchemaBase extends PluginBase implements WebformJsonSchemaInterface, ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The Webform item transformer.
   */
  protected WebformItemTransformer $webformItemTransformer;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->webformItemTransformer = $container->get('webform_headless.transformer.webform_item');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getSchema(WebformInterface $webform): array {
    $items = $this->webformItemTransformer->getWebformItemsFromWebform($webform);
    return $this->buildSchema($webform, $items);
  }

  /**
   * Gets the values from the webform items.
   *
   * @param \Drupal\webform_headless\WebformItem[] $items
   *   The webform items.
   */
  public function getValues(array $items): array {
    $values = [];
    foreach ($items as $key => $item) {
      if ($item->children !== []) {
        $values[$key] = $this->getValues($item->children);
        if ($values[$key] === []) {
          unset($values[$key]);
        }
        continue;
      }

      $value = $item->builtElement['#value'] ?? $item->builtElement['#default_value'] ?? NULL;
      if ($item->elementPlugin instanceof BooleanBase) {
        if ($value) {
          $value = TRUE;
        }
        else {
          // FALSE is a valid value for a required boolean field in Formkit,
          // but not in Webform.
          $value = NULL;
        }
      }
      if ($item->elementPlugin instanceof WebformImageFile) {
        $value = $this->addImagePreviews($value, $item);
      }
      if ($value !== NULL && $value !== '' && $value !== []) {
        $values[$key] = $value;
      }
    }
    return $this->flattenData($values, $items);
  }

  /**
   * {@inheritdoc}
   */
  public function toArray(WebformInterface $webform, Request $request, ?WebformSubmissionInterface $submission = NULL): array {
    $items = $this->webformItemTransformer->getWebformItemsFromWebform($webform, $request, $submission);
    return [
      'schema' => $this->buildSchema($webform, $items),
      'values' => $this->getValues($items),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function normalizeData(array $data, WebformInterface $webform): array {
    $items = $this->webformItemTransformer->getWebformItemsFromWebform($webform);
    return $this->flattenData($data, $items);
  }

  /**
   * {@inheritdoc}
   */
  public function denormalizeData(array $data, WebformInterface $webform): array {
    $items = $this->webformItemTransformer->getWebformItemsFromWebform($webform);
    return $this->unflattenData($data, $items);
  }

  /**
   * Builds the schema.
   *
   * @param \Drupal\webform\WebformInterface $webform
   *   The webform.
   * @param \Drupal\webform_headless\WebformItem[] $items
   *   The webform items.
   */
  abstract protected function buildSchema(WebformInterface $webform, array $items): array;

  /**
   * Prepares JSON data for submission.
   *
   * Webform has two kinds of grouped elements:
   * - composite elements (e.g. address)
   * - field groups (e.g. fieldset)
   *
   * To represent form as JSON schema, we need to treat both kinds in the same
   * way. For example:
   *   [
   *     'address' => [
   *       'street' => {string},
   *       'number' => {string},
   *     ],
   *     'name_fieldset' => [
   *       'first_name' => {string},
   *       'last_name' => {string},
   *     ],
   *   ]
   * This is required to render the form in the proper way on the frontend.
   *
   * But on submission Webform module expects that composite element children
   * are nested on their parent, but field group children are not. Here is how
   * the above example should be passed to
   * WebformSubmissionForm::submitFormValues():
   *   [
   *     'address' => [
   *       'street' => {string},
   *       'number' => {string},
   *     ],
   *     'first_name' => {string},
   *     'last_name' => {string},
   *   ]
   */
  protected function flattenData(array $data, array $items): array {
    $result = [];

    foreach ($data as $key => $value) {
      if (isset($items[$key]) && $items[$key]->children !== [] && is_array($value)) {
        if ($items[$key]->elementPlugin->isComposite() && $items[$key]->elementPlugin instanceof WebformCompositeBase) {
          $result[$key] = self::flattenData($value, $items[$key]->children);
        }
        else {
          $result += self::flattenData($value, $items[$key]->children);
        }
      }
      else {
        $result[$key] = $value;
      }
    }

    return $result;
  }

  /**
   * Prepares submission data for JSON Schema.
   *
   * @see static::flattenData()
   *   To get the logic.
   */
  protected function unflattenData(array $data, array $items): array {
    $result = [];

    foreach ($items as $key => $item) {
      if ($item->children !== []) {
        if ($item->elementPlugin->isComposite() && $item->elementPlugin instanceof WebformCompositeBase) {
          $result += self::unflattenData($data, $item->children);
        }
        else {
          $result[$key] = self::unflattenData($data, $item->children);
        }
      }
      elseif (isset($data[$key])) {
        $result[$key] = $data[$key];
      }
    }

    return $result;
  }

  /**
   * Adds image previews to the value.
   *
   * @param array $value
   *   The value to add previews to.
   * @param \Drupal\webform_headless\WebformItem $item
   *   The webform item.
   *
   * @return array
   *   The value with added previews.
   */
  protected function addImagePreviews(array $value, WebformItem $item): array {
    $fileStorage = $this->entityTypeManager
      ->getStorage('file');

    foreach ($value['fids'] as $fid) {
      $file = $fileStorage->load($fid);
      if (!$file instanceof FileInterface) {
        continue;
      }

      // Don't allow anonymous temporary files to be previewed.
      // @see template_preprocess_file_link().
      // @see webform_preprocess_file_link().
      if ($file->isTemporary() && $file->getOwner()->isAnonymous() && strpos($file->getFileUri(), 'private://') === 0) {
        continue;
      }

      $imageStyleStorage = $this->entityTypeManager->getStorage('image_style');
      [$imageStyleId, $format] = explode(':', $item->element['#file_preview']);

      if ($format === 'image') {
        if (!$imageStyle = $imageStyleStorage->load($imageStyleId)) {
          continue;
        }

        $path = $file->getFileUri();
        if (!$imageStyle->supportsUri($path)) {
          continue;
        }

        $value['previews'][$fid]['format'] = $item->element['#file_preview'];
        $value['previews'][$fid]['value'] = $imageStyle->buildUrl($path);
      }
      else {
        // TODO: Add support for other formats.
      }
    }

    return $value;
  }

}
