<?php

namespace Drupal\webform_headless;

use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormState;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\url_entity\UrlEntityExtractorInterface;
use Drupal\webform\Entity\WebformSubmission;
use Drupal\webform\Plugin\WebformElement\WebformCompositeBase;
use Drupal\webform\WebformInterface;
use Drupal\webform\WebformSubmissionConditionsValidatorInterface;
use Drupal\webform\WebformSubmissionForm;
use Drupal\webform\WebformSubmissionInterface;
use Drupal\webform\WebformTokenManagerInterface;
use Drupal\webform_headless\Exception\WebformSubmissionInvalidException;
use Drupal\webform_headless\Transformer\WebformItemTransformer;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Submits Webform submissions.
 */
class Submitter {

  /**
   * Submitter constructor.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Drupal\Core\Form\FormBuilderInterface $formBuilder
   *   The form builder.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   * @param \Drupal\webform\WebformTokenManagerInterface $tokenManager
   *   The Webform token manager.
   * @param \Drupal\webform_headless\WebformJsonSchemaManager $jsonSchemaManager
   *   The Webform JSON Schema manager.
   * @param \Drupal\webform_headless\Transformer\WebformItemTransformer $webformItemTransformer
   *   The WebformItem transformer.
   * @param \Drupal\url_entity\UrlEntityExtractorInterface $urlEntityExtractor
   *   The URL entity extractor.
   * @param \Drupal\webform\WebformSubmissionConditionsValidatorInterface $conditionsValidator
   *   The Webform submission conditions validator.
   */
  public function __construct(
    protected readonly MessengerInterface $messenger,
    protected readonly FormBuilderInterface $formBuilder,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly LanguageManagerInterface $languageManager,
    protected readonly WebformTokenManagerInterface $tokenManager,
    protected readonly WebformJsonSchemaManager $jsonSchemaManager,
    protected readonly WebformItemTransformer $webformItemTransformer,
    protected readonly UrlEntityExtractorInterface $urlEntityExtractor,
    protected readonly WebformSubmissionConditionsValidatorInterface $conditionsValidator,
  ) {
  }

  /**
   * Creates a new submission.
   */
  public function getSchema(Request $request): WebformJsonSchemaInterface {
    $format = $request->query->get('schema');
    if (!$format) {
      throw new WebformSubmissionInvalidException('Schema format not provided');
    }

    try {
      $schema = $this->jsonSchemaManager->createInstance($format);
    }
    catch (PluginNotFoundException $exception) {
      throw new WebformSubmissionInvalidException(
        message: sprintf('Schema format not found: %s', $format),
        previous: $exception,
      );

    }

    return $schema;
  }

  /**
   * Creates a new submission.
   */
  public function createSubmission(WebformInterface $webform, Request $request): WebformSubmissionInterface {
    $submission = WebformSubmission::create(['webform_id' => $webform->id()]);
    $this->updateSubmission($submission, $request);

    return $submission;
  }

  /**
   * Updates an existing submission.
   */
  public function updateSubmission(WebformSubmissionInterface $submission, Request $request): void {
    $schema = $this->getSchema($request);
    $webform = $submission->getWebform();

    if ($request->getContentTypeFormat() !== 'form') {
      throw new WebformSubmissionInvalidException('Content type must be application/x-www-form-urlencoded or multipart/form-data');
    }

    $isOpen = WebformSubmissionForm::isOpen($webform);
    if ($isOpen !== TRUE) {
      throw new WebformSubmissionInvalidException($isOpen);
    }

    $data = $request->request->all();
    $normalizedData = $schema->normalizeData($data, $webform);

    foreach ($normalizedData as $fieldName => $value) {
      $submission->setElementData($fieldName, $value);
    }

    if ($request->query->has('draft')) {
      $this->checkDraftEnabled($submission);
      $submission->set('in_draft', $request->query->getBoolean('draft'));
      // @todo Add support for current_page.
    }

    $language = $this->languageManager->getCurrentLanguage()->getId();
    $submission->set('langcode', $language);

    if ($submission->isNew()) {
      $refererEntity = $this->urlEntityExtractor->getRefererEntity($request);
      if ($refererEntity) {
        $submission->set('uri', $refererEntity->toUrl()->toString());
        $submission->set('entity_type', $refererEntity->getEntityTypeId());
        $submission->set('entity_id', $refererEntity->id());
      }
    }

    $this->validateWebformSubmission($submission, $data);
    $submission->save();
  }

  /**
   * Returns an error response from the result of submission.
   */
  protected function getErrorResponse(WebformInterface $webform, array $errors): JsonResponse {
    $items = $this->webformItemTransformer->getWebformItemsFromWebform($webform);
    $mapping = $this->getPathMapping($items);
    $normalizedErrors = [];

    foreach ($errors as $formPath => $error) {
      $mappedPath = $mapping[$formPath] ?? $formPath;
      $path = explode('][', $mappedPath);
      $normalizedErrors[] = [
        'message' => strip_tags((string) $error),
        'path' => $path,
      ];
    }

    return new JsonResponse(
      ['errors' => $normalizedErrors],
      Response::HTTP_BAD_REQUEST
    );
  }

  /**
   * Returns path mapping for Webform error messages.
   *
   * @see static::flattenData()
   *   To get the logic.
   *
   * @return array
   *   The mapping array. Keys are form paths, values are data paths.
   */
  protected function getPathMapping(array $items, array $json_path = [], array $webform_path = [], array &$result = []): array {
    foreach ($items as $key => $item) {
      if ($item->children !== []) {
        $json_path[] = $key;
        if ($item->elementPlugin->isComposite() && $item->elementPlugin instanceof WebformCompositeBase) {
          $webform_path[] = $key;
          $this->getPathMapping($item->children, $json_path, $webform_path, $result);
          array_pop($webform_path);
        }
        else {
          $this->getPathMapping($item->children, $json_path, $webform_path, $result);
        }
        array_pop($json_path);
      }
      else {
        if ($json_path !== $webform_path) {
          $result[implode('][', array_merge($webform_path, [$key]))] = implode('][', array_merge($json_path, [$key]));
        }
      }
    }

    return $result;
  }

  /**
   * Returns the confirmation settings for a Webform.
   */
  public function getConfirmationSettings(WebformSubmissionInterface $submission): array {
    $webform = $submission->getWebform();

    $data = [];
    $data['type'] = $confirmationType = $webform->getSetting('confirmation_type', TRUE);
    $data['title'] = $webform->getSetting('confirmation_title', TRUE);
    $data['title'] = $this->tokenManager->replace($data['title'], $submission);

    if (in_array($confirmationType, [WebformInterface::CONFIRMATION_URL, WebformInterface::CONFIRMATION_URL_MESSAGE])) {
      $data['url'] = $webform->getSetting('confirmation_url', TRUE);
      $data['url'] = $this->tokenManager->replace($data['url'], $submission);
    }

    if (in_array($confirmationType, [WebformInterface::CONFIRMATION_PAGE, WebformInterface::CONFIRMATION_URL, WebformInterface::CONFIRMATION_URL_MESSAGE])) {
      $data['excludeQuery'] = $webform->getSetting('confirmation_exclude_query', TRUE);
      $data['excludeToken'] = $webform->getSetting('confirmation_exclude_token', TRUE);
    }

    if (!in_array($confirmationType, [WebformInterface::CONFIRMATION_URL, WebformInterface::CONFIRMATION_NONE])) {
      $message = $webform->getSetting('confirmation_message', TRUE);
      $data['message'] = $this->tokenManager->replace($message, $submission);
    }

    if (in_array($confirmationType, [WebformInterface::CONFIRMATION_PAGE, WebformInterface::CONFIRMATION_INLINE])) {
      $data['attributes'] = $webform->getSetting('confirmation_attributes', TRUE);
      $data['back'] = $webform->getSetting('confirmation_back', TRUE);
      $data['backLabel'] = $webform->getSetting('confirmation_back_label', TRUE);
      $data['backAttributes'] = $webform->getSetting('confirmation_back_attributes', TRUE);
    }

    $data['update'] = $webform->getSetting('confirmation_update', TRUE);

    return $data;
  }

  /**
   * Programmatically validate a webform submission.
   *
   * @param \Drupal\webform\WebformSubmissionInterface $submission
   *   The webform submission.
   * @param array $data
   *   The original data.
   *
   * @throws \Drupal\webform_headless\Exception\WebformSubmissionInvalidException
   *
   * @see \Drupal\webform\WebformSubmissionForm::submitWebformSubmission()
   */
  protected function validateWebformSubmission(WebformSubmissionInterface $submission, array $data): void {
    $formObject = $this->entityTypeManager->getFormObject('webform_submission', 'api');
    assert($formObject instanceof WebformSubmissionForm);
    $formObject->setEntity($submission);

    // Create an empty form state which will be populated when the submission
    // form is submitted.
    $formState = new FormState();
    // @see https://www.drupal.org/project/webform/issues/3469703
    $formState->setValues($data);

    // Set the triggering element to an empty element to prevent
    // errors from managed files.
    // @see \Drupal\file\Element\ManagedFile::validateManagedFile
    $formState->setTriggeringElement(['#parents' => []]);

    // Get existing error messages.
    $errorMessages = $this->messenger->messagesByType(MessengerInterface::TYPE_ERROR);

    // Submit the form.
    $this->formBuilder->submitForm($formObject, $formState);

    // Get the errors but skip drafts.
    $errors = $submission->isDraft() ? [] : $formState->getErrors();

    // Delete all form related error messages.
    $this->messenger->deleteByType(MessengerInterface::TYPE_ERROR);

    // Restore existing error message.
    foreach ($errorMessages as $error_message) {
      $this->messenger->addError($error_message);
    }

    // Filter out incorrect required errors.
    $errors = $this->filterIncorrectRequiredErrors($submission, $errors);

    if ($errors !== []) {
      throw (new WebformSubmissionInvalidException())->setErrors($errors);
    }
  }

  /**
   * Filters out incorrect required errors.
   *
   * This is a workaround for an issue with the Webform module.
   *
   * @param \Drupal\webform\WebformSubmissionInterface $submission
   *   The webform submission.
   * @param array $errors
   *   The errors.
   *
   * @return array
   *   The filtered errors.
   *
   * @see https://www.drupal.org/project/webform/issues/3469703
   */
  protected function filterIncorrectRequiredErrors(WebformSubmissionInterface $submission, array $errors): array {
    $webform = $submission->getWebform();
    if (!$webform instanceof WebformInterface) {
      return $errors;
    }

    $data = $submission->getData();

    foreach ($errors as $key => $error) {
      if ($error instanceof TranslatableMarkup) {
        $error = $error->getUntranslatedString();
      }

      $element = $webform->getElement($key);
      $hasFormRequiredError = (is_string($error) && preg_match('/^.+ field is required\.$/', $error))
       || (is_array($element) && isset($element['#required_error']) && $element['#required_error'] === $error);

      if (!$hasFormRequiredError) {
        continue;
      }

      $hasValue = isset($data[$key]) && $data[$key] !== '' && $data[$key] !== [];
      $isHidden = array_reduce(
        $element['#webform_parents'],
        fn ($carry, $parent) => $carry || !$this->conditionsValidator->isElementVisible($webform->getElement($parent), $submission),
      );

      if ($hasValue || $isHidden) {
        unset($errors[$key]);
      }
    }

    return $errors;
  }

  /**
   * Determine if drafts are enabled.
   *
   * @see \Drupal\webform\WebformSubmissionForm::draftEnabled()
   */
  protected function checkDraftEnabled(WebformSubmissionInterface $submission): void {
    $webform = $submission->getWebform();

    // Can't saved drafts when saving results is disabled.
    if ($webform->getSetting('results_disabled')) {
      throw new WebformSubmissionInvalidException('Draft submissions are not allowed for this webform because saving of submissions is disabled.');
    }

    // Once a form is completed drafts are no longer applicable.
    if ($submission->isCompleted()) {
      throw new WebformSubmissionInvalidException('This submission cannot be saved as draft because it is already completed.');
    }

    if ($webform->getSetting('draft') === WebformInterface::DRAFT_AUTHENTICATED && !$submission->getOwner()->isAuthenticated()) {
      throw new WebformSubmissionInvalidException('Draft submissions are not allowed for this webform because the user is not authenticated.');
    }
  }

}
