<?php

namespace Drupal\webform_headless\Transformer;

use Drupal\Core\Entity\EntityFormBuilderInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Exception\UndefinedLinkTemplateException;
use Drupal\Core\Render\Element;
use Drupal\url_entity\UrlEntityExtractorInterface;
use Drupal\webform\Plugin\WebformElement\WebformCompositeBase;
use Drupal\webform\Plugin\WebformElementInterface;
use Drupal\webform\Plugin\WebformElementManagerInterface;
use Drupal\webform\WebformInterface;
use Drupal\webform\WebformSubmissionInterface;
use Drupal\webform_headless\Exception\WebformSubmissionForbiddenException;
use Drupal\webform_headless\WebformItem;
use Symfony\Component\HttpFoundation\Request;

/**
 * Transforms a webform into a JSON schema.
 */
class WebformItemTransformer {

  /**
   * Constructs a new WebformItemTransformer.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFormBuilderInterface $entityFormBuilder
   *   The entity form builder.
   * @param \Drupal\webform\Plugin\WebformElementManagerInterface $elementManager
   *   The Webform element manager.
   * @param \Drupal\url_entity\UrlEntityExtractorInterface $urlEntityExtractor
   *   The URL entity extractor.
   */
  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly EntityFormBuilderInterface $entityFormBuilder,
    protected readonly WebformElementManagerInterface $elementManager,
    protected readonly UrlEntityExtractorInterface $urlEntityExtractor,
  ) {
  }

  /**
   * Gets the webform items from a webform.
   *
   * @param \Drupal\webform\WebformInterface $webform
   *   The webform entity.
   * @param \Symfony\Component\HttpFoundation\Request|null $request
   *   The request, used to extract the referer entity.
   * @param \Drupal\webform\WebformSubmissionInterface|null $submission
   *   The webform submission, if available. If not provided, a new submission
   *   will be created with the webform's ID and referer entity (if available).
   *
   * @return \Drupal\webform_headless\WebformItem[]
   *   The webform items.
   */
  public function getWebformItemsFromWebform(WebformInterface $webform, ?Request $request = NULL, ?WebformSubmissionInterface $submission = NULL): array {
    if (!$submission instanceof WebformSubmissionInterface) {
      $values = [];
      // Set this webform's id which can be used by preCreate hooks.
      $values['webform_id'] = $webform->id();

      if ($request instanceof Request) {
        $refererEntity = $this->urlEntityExtractor->getRefererEntity($request);
        if ($refererEntity && !$refererEntity->isNew()) {
          try {
            $values['uri'] = $refererEntity->toUrl()->toString();
          } catch (UndefinedLinkTemplateException) {
            // Ignore the exception.
          }
          $values['entity_type'] = $refererEntity->getEntityTypeId();
          $values['entity_id'] = $refererEntity->id();
        }
      }

      $submission = $this->entityTypeManager
        ->getStorage('webform_submission')
        ->create($values);
      assert($submission instanceof WebformSubmissionInterface);
    }

    $operation = $submission->isNew() ? 'add' : 'edit';
    $form = $this->entityFormBuilder->getForm($submission, $operation);

    if (!isset($form['elements'])) {
      $exception = new WebformSubmissionForbiddenException();
      if (isset($form['webform_message'])) {
        foreach (Element::children($form['webform_message']) as $elementKey) {
          if (isset($form['webform_message'][$elementKey]['#message_message']['#markup'])) {
            $exception->addError($form['webform_message'][$elementKey]['#message_message']['#markup']);
          }
        }
      }
      throw $exception;
    }

    return $this->getWebformItemsFromElements($form['elements'], $webform);
  }

  /**
   * Gets a webform item from a webform.
   *
   * @return \Drupal\webform_headless\WebformItem|null
   *   The webform item or null if the element does not exist/is not accessible.
   */
  public function getWebformItemFromWebform(WebformInterface $webform, string $key): ?WebformItem {
    $element = $webform->getElement($key);
    if ($element === NULL) {
      return NULL;
    }

    return $this->getWebformItemFromElement($element, $webform);
  }

  /**
   * Gets a webform item from an element.
   *
   * @return \Drupal\webform_headless\WebformItem|null
   *   The webform item, or null if the element is not accessible.
   */
  protected function getWebformItemFromElement(array $element, WebformInterface $webform): ?WebformItem {
    $elementPlugin = $this->elementManager->getElementInstance($element, $webform);
    assert($elementPlugin instanceof WebformElementInterface);

    $item = new WebformItem();
    if (isset($element['#webform_key']) && $initializedElement = $webform->getElementInitialized($element['#webform_key'])) {
      $item->element = $initializedElement;
    }
    else {
      $item->element = $element;
    }
    $item->builtElement = $element;
    $item->elementPlugin = $elementPlugin;

    if ($elementPlugin->isComposite() && $elementPlugin instanceof WebformCompositeBase) {
      $childElements = $elementPlugin->getInitializedCompositeElement($element);
      $childItems = $this->getWebformItemsFromElements($childElements, $webform);
      foreach ($childItems as $childKey => $childItem) {
        $parentKey = $childItem->element['#webform_composite_parent_key'];
        $item->children["{$parentKey}[{$childKey}]"] = $childItem;
      }
    }
    elseif ($elementPlugin->isContainer($element) && $childrenKeys = Element::children($element)) {
      $children = array_intersect_key($element, array_flip($childrenKeys));
      $item->children = $this->getWebformItemsFromElements($children, $webform);
    }

    if (!isset($originalElement['#access']) || $originalElement['#access']) {
      return $item;
    }

    return NULL;
  }

  /**
   * Gets the webform items from elements.
   *
   * @return \Drupal\webform_headless\WebformItem[]
   *   The webform items.
   */
  protected function getWebformItemsFromElements(array $elements, WebformInterface $webform): array {
    $items = [];

    foreach (Element::children($elements) as $key) {
      $element = $elements[$key];
      $item = $this->getWebformItemFromElement($element, $webform);
      if ($item !== NULL) {
        $items[$key] = $item;
      }
    }

    return $items;
  }

}
