<?php

namespace Drupal\webform_headless;

use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\webform_headless\Annotation\WebformJsonSchema;
use Drupal\webform_headless\Attribute\WebformJsonSchema as WebformJsonSchemaAttribute;
use Drupal\webform_headless\Exception\WebformSubmissionInvalidException;
use Symfony\Component\HttpFoundation\Request;

/**
 * The WebformJsonSchema plugin manager.
 *
 * @method \Drupal\webform_headless\WebformJsonSchemaInterface createInstance($plugin_id, array $configuration = [])
 */
class WebformJsonSchemaManager extends DefaultPluginManager {

  /**
   * {@inheritdoc}
   */
  public function __construct(
    \Traversable $namespaces,
    CacheBackendInterface $cache_backend,
    ModuleHandlerInterface $module_handler,
  ) {
    parent::__construct(
      'Plugin/WebformJsonSchema',
      $namespaces,
      $module_handler,
      WebformJsonSchemaInterface::class,
      WebformJsonSchemaAttribute::class,
      WebformJsonSchema::class
    );

    $this->alterInfo('webform_headless_json_schema');
    $this->setCacheBackend($cache_backend, 'webform_headless_json_schema');
  }

  /**
   * Gets a schema plugin by request.
   */
  public function getByRequest(Request $request): WebformJsonSchemaInterface {
    $format = $request->query->get('schema');
    if (!$format) {
      throw new WebformSubmissionInvalidException('Schema format not provided');
    }

    return $this->getByFormat($format);
  }

  /**
   * Gets a schema plugin by format.
   */
  public function getByFormat(string $format): WebformJsonSchemaInterface {
    try {
      $schema = $this->createInstance($format);
    }
    catch (PluginNotFoundException $exception) {
      $format = mb_convert_encoding($format, 'UTF-8', 'UTF-8');
      throw new WebformSubmissionInvalidException(
        message: sprintf('Schema format not found: %s', $format),
        previous: $exception,
      );
    }

    return $schema;
  }

}
