<?php

namespace Drupal\webform_mailerlite\Plugin\WebformHandler;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Site\Settings;
use Drupal\webform\Plugin\WebformHandlerBase;
use Drupal\webform\WebformSubmissionInterface;
use MailerLite\MailerLite;
use MailerLite\Exceptions\MailerLiteException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form submission to MailerLite handler.
 *
 * @WebformHandler(
 *   id = "mailerlite",
 *   label = @Translation("MailerLite"),
 *   category = @Translation("MailerLite"),
 *   description = @Translation("Sends a form submission to a MailerLite list."),
 *   cardinality = \Drupal\webform\Plugin\WebformHandlerInterface::CARDINALITY_UNLIMITED,
 *   results = \Drupal\webform\Plugin\WebformHandlerInterface::RESULTS_PROCESSED,
 * )
 */
class WebformMailerliteHandler extends WebformHandlerBase {

  /**
   * The Mailerlite client.
   *
   * @var \MailerLite\MailerLite
   */
  protected MailerLite $mailerlite;

  /**
   * The token manager.
   *
   * @var \Drupal\webform\WebformTokenManagerInterface
   */
  protected $tokenManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->tokenManager = $container->get('webform.token_manager');
    $key = Settings::get("mailerlite_key", "");
    $instance->mailerlite = new MailerLite(['api_key' => $key]);

    return $instance;
  }

  /**
   * Get MailerLite groups as options array.
   *
   * @return array
   *   Array of group options keyed by group ID.
   */
  protected function getGroupOptions() {
    $options = [];

    try {
      $groupsData = $this->mailerlite->groups->get();

      if (isset($groupsData['body']['data']) && is_array($groupsData['body']['data'])) {
        foreach ($groupsData['body']['data'] as $group) {
          if (isset($group['id']) && isset($group['name'])) {
            $options[$group['id']] = $group['name'];
          }
        }
      }
    }
    catch (MailerLiteException $e) {
      \Drupal::logger('webform_mailerlite')->error(
        'Error fetching MailerLite groups: @error',
        ['@error' => $e->getMessage()]
      );
    }
    catch (\Exception $e) {
      \Drupal::logger('webform_mailerlite')->error(
        'General error fetching MailerLite groups: @error',
        ['@error' => $e->getMessage()]
      );
    }

    return $options;
  }

  /**
   * Get group name by ID for summary display.
   *
   * @param string $group_id
   *   The group ID.
   *
   * @return string
   *   The group name or ID if name not found.
   */
  protected function getGroupName($group_id) {
    if (empty($group_id)) {
      return 'None';
    }

    $options = $this->getGroupOptions();
    return isset($options[$group_id]) ? $options[$group_id] : $group_id;
  }

  /**
   * {@inheritdoc}
   */
  public function getSummary() {
    $key = Settings::get("mailerlite_key", "");
    $fields = $this->getWebform()->getElementsInitializedAndFlattened();

    $key = substr($key, 0, 10);
    $key_summary = '<strong>' . $this->t('Key') . ': </strong>' . $key;

    $email_summary = $this->configuration['email'];
    if (!empty($fields[$this->configuration['email']])) {
      $email_summary = $fields[$this->configuration['email']]['#title'];
    }
    $email_summary = '<strong>' . $this->t('Email') . ': </strong>' . $email_summary;

    // Add group summary with group name.
    $group_name = $this->getGroupName($this->configuration['group_id'] ?? '');
    $group_summary = '<strong>' . $this->t('Group') . ': </strong>' . $group_name;

    // Add fields mapping summary.
    $fields_mapping = '';
    if (!empty($this->configuration['fields'])) {
      $mapping_lines = explode("\n", trim($this->configuration['fields']));
      $mapped_fields = [];
      foreach ($mapping_lines as $line) {
        $line = trim($line);
        if (!empty($line) && strpos($line, ':') !== FALSE) {
          $field_parts = explode(':', $line, 2);
          if (count($field_parts) == 2) {
            $webform_field = trim($field_parts[0]);
            $mailerlite_field = trim($field_parts[1]);

            // Get the webform field title if available.
            $field_title = $webform_field;
            if (!empty($fields[$webform_field]['#title'])) {
              $field_title = $fields[$webform_field]['#title'];
            }

            $mapped_fields[] = "{$field_title} → {$mailerlite_field}";
          }
        }
      }

      if (!empty($mapped_fields)) {
        $fields_mapping = '<strong>' . $this->t('Extra Field Mapping') . ': </strong><br>' . implode('<br>', $mapped_fields);
      }
      else {
        $fields_mapping = '<strong>' . $this->t('Extra Field Mapping') . ': </strong>' . $this->t('None configured');
      }
    }
    else {
      $fields_mapping = '<strong>' . $this->t('Extra Field Mapping') . ': </strong>' . $this->t('None configured');
    }

    $markup = "$key_summary<hr>$email_summary<hr>$group_summary<hr>$fields_mapping";
    return [
      '#markup' => $markup,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'list' => '',
      'email' => '',
      'fields' => [],
      'group_id' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $enabled = $this->configFactory->get('mailerlite.enabled_lists')->getRawData();

    $form['mailerlite'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('MailerLite settings'),
      '#attributes' => ['id' => 'webform-mailerlite-handler-settings'],
    ];

    $fields = $this->getWebform()->getElementsInitializedAndFlattened();
    $options = [];
    foreach ($fields as $field_name => $field) {
      if (in_array($field['#type'], ['email', 'webform_email_confirm'])) {
        $options[$field_name] = $field['#title'];
      }
    }

    $default_value = $this->configuration['email'];
    if (empty($this->configuration['email']) && count($options) == 1) {
      $default_value = key($options);
    }
    $form['mailerlite']['email'] = [
      '#type' => 'webform_select_other',
      '#title' => $this->t('Email field'),
      '#required' => TRUE,
      '#default_value' => $default_value,
      '#options' => $options,
      '#empty_option' => $this->t('- Select an option -'),
      '#description' => $this->t('Select the email element you want to use for subscribing to the MailerLite list specified above.'),
    ];

    // Get MailerLite groups for dropdown.
    $group_options = $this->getGroupOptions();

    // Add group selection dropdown.
    $form['mailerlite']['group_id'] = [
      '#type' => 'select',
      '#title' => $this->t('MailerLite Group'),
      '#default_value' => $this->configuration['group_id'] ?? '',
      '#options' => $group_options,
      '#empty_option' => $this->t('- No group (create subscriber without group) -'),
      '#description' => $this->t('Select the MailerLite group where subscribers should be added. Leave empty to create subscriber without group assignment.'),
    ];

    // Add a refresh button to reload groups if needed.
    if (empty($group_options)) {
      $form['mailerlite']['group_refresh_message'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--warning">' .
        $this->t('No MailerLite groups found. Please check your API key configuration or ensure you have groups created in your MailerLite account.') .
        '</div>',
      ];
    }

    $form['mailerlite']['fields'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Fields Mapping'),
      '#description' => $this->t('Enter the fields you want to map to MailerLite. Each field should be on a new line in the format "webform_field_name: mailerlite_field_name".'),
      '#default_value' => $this->configuration['fields'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    $values = $form_state->getValues();
    foreach ($this->configuration as $name => $value) {
      if (isset($values['mailerlite'][$name])) {
        $this->configuration[$name] = $values['mailerlite'][$name];
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(WebformSubmissionInterface $webform_submission, $update = TRUE) {
    // Process both new submissions and updates.
    $fields = $webform_submission->toArray(TRUE);
    $configuration = $this->tokenManager->replace($this->configuration, $webform_submission);

    // Email could be a webform element or a string/token.
    if (!empty($fields['data'][$configuration['email']])) {
      $email = $fields['data'][$configuration['email']];
    }
    else {
      $email = $configuration['email'];
    }

    if (!empty($email)) {
      try {
        $data = [
          'email' => $email,
        ];

        // Add custom fields mapping.
        if (!empty($configuration['fields'])) {
          $configurationFields = explode("\n", $configuration['fields']);
          foreach ($configurationFields as $field) {
            $field_parts = explode(':', $field, 2);
            if (count($field_parts) == 2) {
              $webform_field = trim($field_parts[0]);
              $mailerlite_field = trim($field_parts[1]);

              if (!empty($webform_field) && !empty($mailerlite_field)) {
                $field_value = '';
                if (isset($fields['data'][$webform_field])) {
                  $field_value = $fields['data'][$webform_field];
                  // Handle array values (like checkboxes)
                  if (is_array($field_value)) {
                    $field_value = implode(', ', array_filter($field_value));
                  }
                }
                $data['fields'][$mailerlite_field] = (string) $field_value;
              }
            }
          }
        }

        $groupId = trim($configuration['group_id'] ?? '');
        $action_type = $update ? 'updated' : 'created';

        // Debug: Log the submission details.
        \Drupal::logger('webform_mailerlite')->info(
          'Processing @action_type submission - Email: @email, Group ID: @group_id, Fields: @fields',
          [
            '@action_type' => $action_type,
            '@email' => $email,
            '@group_id' => $groupId ?: 'NONE',
            '@fields' => json_encode($data['fields'] ?? []),
          ]
        );

        // Try to update existing subscriber first, then create if not found.
        $subscriber_exists = FALSE;
        $subscriber_id = NULL;

        try {
          // Check if subscriber exists by getting subscriber info.
          $existing_subscriber = $this->mailerlite->subscribers->find($email);
          if (isset($existing_subscriber['data']['id'])) {
            $subscriber_exists = TRUE;
            $subscriber_id = $existing_subscriber['data']['id'];

            \Drupal::logger('webform_mailerlite')->info(
              'Existing subscriber found: @email (ID: @id)',
              ['@email' => $email, '@id' => $subscriber_id]
            );
          }
        }
        catch (MailerLiteException $e) {
          // Subscriber doesn't exist or other API error.
          \Drupal::logger('webform_mailerlite')->info(
            'Subscriber not found or API error when checking: @error',
            ['@error' => $e->getMessage()]
          );
        }

        if ($subscriber_exists && $subscriber_id) {
          // Update existing subscriber.
          try {
            $update_data = [];
            if (isset($data['fields']) && !empty($data['fields'])) {
              $update_data['fields'] = $data['fields'];
            }

            if (!empty($update_data)) {
              $updated_subscriber = $this->mailerlite->subscribers->update($subscriber_id, $update_data);

              \Drupal::logger('webform_mailerlite')->info(
                'Subscriber @email updated successfully with fields: @fields',
                [
                  '@email' => $email,
                  '@fields' => json_encode($update_data['fields'] ?? []),
                ]
              );
            }

            // Handle group assignment for existing subscriber.
            if (!empty($groupId)) {
              try {
                $assign_result = $this->mailerlite->groups->assignSubscriber($groupId, $subscriber_id);

                \Drupal::logger('webform_mailerlite')->info(
                  'Existing subscriber @email assigned to group @group_id',
                  ['@email' => $email, '@group_id' => $groupId]
                );
              }
              catch (MailerLiteException $assign_e) {
                \Drupal::logger('webform_mailerlite')->warning(
                  'Could not assign existing subscriber to group: @error',
                  ['@error' => $assign_e->getMessage()]
                );
              }
            }
          }
          catch (MailerLiteException $e) {
            \Drupal::logger('webform_mailerlite')->error(
              'Error updating existing subscriber @email: @error',
              ['@email' => $email, '@error' => $e->getMessage()]
            );
          }
        }
        else {
          // Create new subscriber.
          if (!empty($groupId)) {
            try {
              // Method 1: Create subscriber with groups array.
              $dataWithGroups = $data;
              $dataWithGroups['groups'] = [$groupId];

              $subscriber = $this->mailerlite->subscribers->create($dataWithGroups);

              \Drupal::logger('webform_mailerlite')->info(
                'New subscriber @email created and added to group @group_id with fields: @fields',
                [
                  '@email' => $email,
                  '@group_id' => $groupId,
                  '@fields' => json_encode($data['fields'] ?? []),
                ]
              );
            }
            catch (MailerLiteException $e) {
              \Drupal::logger('webform_mailerlite')->warning(
                'Method 1 failed, trying Method 2: @error',
                ['@error' => $e->getMessage()]
              );

              // Method 2: Create subscriber first, then assign to group.
              try {
                $subscriber = $this->mailerlite->subscribers->create($data);

                if (isset($subscriber['data']['id'])) {
                  $new_subscriber_id = $subscriber['data']['id'];

                  $assign_result = $this->mailerlite->groups->assignSubscriber($groupId, $new_subscriber_id);

                  \Drupal::logger('webform_mailerlite')->info(
                    'New subscriber @email created and assigned to group @group_id via Method 2 with fields: @fields',
                    [
                      '@email' => $email,
                      '@group_id' => $groupId,
                      '@fields' => json_encode($data['fields'] ?? []),
                    ]
                  );
                }
                else {
                  \Drupal::logger('webform_mailerlite')->error(
                    'No subscriber ID returned from MailerLite'
                  );
                }
              }
              catch (MailerLiteException $assign_e) {
                \Drupal::logger('webform_mailerlite')->error(
                  'Method 2 also failed - Cannot assign subscriber to group: @error',
                  ['@error' => $assign_e->getMessage()]
                );

                // Still log that subscriber was created without group.
                \Drupal::logger('webform_mailerlite')->info(
                  'New subscriber @email created without group assignment due to error with fields: @fields',
                  [
                    '@email' => $email,
                    '@fields' => json_encode($data['fields'] ?? []),
                  ]
                );
              }
            }
          }
          else {
            // No group ID specified - create subscriber without group.
            $subscriber = $this->mailerlite->subscribers->create($data);

            \Drupal::logger('webform_mailerlite')->info(
              'New subscriber @email created without group assignment with fields: @fields',
              [
                '@email' => $email,
                '@fields' => json_encode($data['fields'] ?? []),
              ]
            );
          }
        }

      }
      catch (MailerLiteException $e) {
        \Drupal::logger('webform_mailerlite')->error(
          'MailerLite API error: @error. Email: @email, Group ID: @group_id, Action: @action',
          [
            '@error' => $e->getMessage(),
            '@email' => $email,
            '@group_id' => $configuration['group_id'] ?? 'EMPTY',
            '@action' => $action_type,
          ]
        );
      }
      catch (\Exception $e) {
        \Drupal::logger('webform_mailerlite')->error(
          'General error processing subscriber in MailerLite: @error. Email: @email, Action: @action',
          [
            '@error' => $e->getMessage(),
            '@email' => $email,
            '@action' => $action_type,
          ]
        );
      }
    }
    else {
      \Drupal::logger('webform_submission')->warning(
        'webform_mailerlite: No email address was provided to the handler.',
      );
    }
  }

}
