<?php

declare(strict_types=1);

namespace Drupal\webform_no_submissions\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\RedundantEditableConfigNamesTrait;
use Drupal\Core\State\StateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Webform No Submissions settings for this site.
 *
 * This form is based on the core Site Maintenance Mode form.
 *
 * @see \Drupal\system\Form\SiteMaintenanceModeForm
 */
class WebformNoSubmissionsSettingsForm extends ConfigFormBase {
  use RedundantEditableConfigNamesTrait;

  /**
   * The state key/value collection.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs a new SiteMaintenanceModeForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state keyvalue collection to use.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typedConfigManager, StateInterface $state) {
    parent::__construct($config_factory, $typedConfigManager);
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('state'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'webform_no_submissions_webform_no_submissions_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['webform_no_submissions'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable all webforms that save submissions to the database'),
      '#default_value' => $this->state->get('webform_no_submissions'),
      '#description' => $this->t('Users will not be allowed to submit webforms that save submissions to the data base but webforms that route submissions to email or APIs will remain available.'),
    ];
    $form['message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Message to display when a webform is disabled'),
      '#config_target' => 'webform_no_submissions.settings:message',
      '#description' => $this->t('Enter the message to be displayed when a webform is disabled.'),
      '#required' => TRUE,
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->state->set('webform_no_submissions', $form_state->getValue('webform_no_submissions'));
    parent::submitForm($form, $form_state);
  }

}
