<?php

declare(strict_types=1);

namespace Drupal\Tests\webform_no_submissions\Functional;

use Drupal\Core\Cache\Cache;
use Drupal\Tests\BrowserTestBase;
use Drupal\webform\Entity\Webform;

/**
 * Test Webform no submissions behavior.
 *
 * @group webform_no_submissions
 */
class WebformNoSubmissionsTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['webform_no_submissions'];

  /**
   * Test Webform no submissions behavior.
   */
  public function testWebformNoSubmissions(): void {
    /** @var \Drupal\webform\Entity\WebformInterface $webform */
    $webform = Webform::load('contact');
    $assert = $this->assertSession();

    // Check that the contact form is enabled.
    $this->drupalGet('/webform/contact');
    $assert->fieldExists('name');

    // Login as root user and enable webform no submissions.
    $this->drupalLogin($this->rootUser);
    $this->drupalGet('/admin/config/development/webform-no-submissions');
    $this->submitForm(['webform_no_submissions' => TRUE], 'Save configuration');

    // Check that the webform no submissions enabled message is displayed on all webform routes for admins.
    $this->drupalGet('/admin/structure/webform');
    $assert->responseContains('All webforms that save submissions to the database are disabled.');
    $this->drupalGet('/webform/contact');
    $assert->responseContains('All webforms that save submissions to the database are disabled.');
    $this->drupalGet('<front>');
    $assert->responseNotContains('All webforms that save submissions to the database are disabled.');

    // Logout as root user and check no submissions for anonymous user.
    $this->drupalLogout();

    // Check that the webform no submissions enabled message is not displayed for an anonymous user.
    $this->drupalGet('/webform/contact');
    $assert->responseNotContains('All webforms that save submissions to the database are disabled.');

    // Check that the contact form is disabled.
    $this->drupalGet('/webform/contact');
    $assert->fieldNotExists('name');
    $assert->responseContains('This webform is unavailable. We should be back shortly. Thank you for your patience.');

    // Check that the contact form is disabled.
    $this->drupalGet('/webform/contact');
    $assert->fieldNotExists('name');
    $assert->responseContains('This webform is unavailable. We should be back shortly. Thank you for your patience.');

    // Disable the saving of submissions for the contact form.
    $webform->setSetting('results_disabled', TRUE)->save();

    // Check that the contact form is enabled.
    $this->drupalGet('/webform/contact');
    $assert->fieldExists('name');
    $assert->responseNotContains('This webform is unavailable. We should be back shortly. Thank you for your patience.');

    // Enable the saving of submissions for the contact form.
    $webform->setSetting('results_disabled', FALSE)->save();

    // Check that the contact form is disabled.
    $this->drupalGet('/webform/contact');
    $assert->fieldNotExists('name');
    $assert->responseContains('This webform is unavailable. We should be back shortly. Thank you for your patience.');

    // Enable webform no submissions via state
    // and invalidate webform_no_submissions.settings config object cache.
    \Drupal::state()->set('webform_no_submissions', FALSE);
    Cache::invalidateTags(['config:webform_no_submissions.settings']);

    // Check that the contact form is re-enabled.
    $this->drupalGet('/webform/contact');
    $assert->fieldExists('name');
    $assert->responseNotContains('This webform is unavailable. We should be back shortly. Thank you for your patience.');
  }

}
