<?php

declare(strict_types=1);

namespace Drupal\webform_openpostcode\Ajax;

use Drupal\Component\Utility\Html;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Form\FormStateInterface;
use Drupal\webform_openpostcode\Service\OpenPostcodeLookupInterface;

/**
 * Ajax callback for postcode lookups.
 */
final class PostcodeLookupAjax {

  /**
   * Ajax callback handler for postcode lookups.
   *
   * @param array $form
   *   The form render array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The Ajax response.
   */
  public static function handle(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();

    $trigger = $form_state->getTriggeringElement();
    $config = $trigger['#webform_openpostcode'] ?? NULL;
    if (!is_array($config) || empty($config['postcode_key']) || empty($config['number_key'])) {
      return $response;
    }

    $submitted = $form_state->getValues();
    $postcode = (string) ($submitted[$config['postcode_key']] ?? '');
    $houseNumber = (string) ($submitted[$config['number_key']] ?? '');

    if ($postcode === '' || $houseNumber === '') {
      return $response;
    }

    try {
      /** @var \Drupal\webform_openpostcode\Service\OpenPostcodeLookupInterface $lookup */
      $lookup = \Drupal::service(OpenPostcodeLookupInterface::class);
      $data = $lookup->lookup($postcode, $houseNumber);
    }
    catch (\InvalidArgumentException $exception) {
      return $response->addCommand(new MessageCommand(t('Postcode lookup requires both postcode and house number.')));
    }
    catch (\Throwable $exception) {
      return $response->addCommand(new MessageCommand(t('Postcode lookup failed. Please try again.')));
    }

    self::applyValueCommand($response, $submitted, $config['street_target_key'] ?? '', (string) ($data['straat'] ?? ''));
    self::applyValueCommand($response, $submitted, $config['city_target_key'] ?? '', (string) ($data['woonplaats'] ?? ''));

    // When the lookup was successful, we'll neatly format the postcode as well,
    // according to the Dutch postal code format, which is 4 digits, a space and
    // upper case letters.
    $postcode = strtoupper(str_replace(' ', '', trim($postcode)));
    $digits = substr($postcode, 0, 4);
    $letters = substr($postcode, 4);
    $postcode = $digits . ' ' . $letters;

    self::applyValueCommand($response, $submitted, $config['postcode_key'] ?? '', $postcode, TRUE);
    return $response;
  }

  /**
   * Add Ajax commands that populate a target input.
   *
   * @param \Drupal\Core\Ajax\AjaxResponse $response
   *   The Ajax response.
   * @param array $submitted
   *   Current submitted values.
   * @param string $targetKey
   *   The target element key.
   * @param string $value
   *   The value to apply.
   * @param bool $overwrite
   *   Whether to overwrite when the field is not empty. Default is false.
   */
  private static function applyValueCommand(AjaxResponse $response, array $submitted, string $targetKey, string $value, bool $overwrite = FALSE): void {
    if ($targetKey === '' || $value === '') {
      return;
    }

    $currentValue = (string) ($submitted[$targetKey] ?? '');
    if ($currentValue !== '' && !$overwrite) {
      return;
    }

    $selector = sprintf(':input[name="%s"]', Html::escape($targetKey));
    $response->addCommand(new InvokeCommand($selector, 'val', [$value]));
    $response->addCommand(new InvokeCommand($selector, 'trigger', ['change']));
  }

}
