<?php

declare(strict_types=1);

namespace Drupal\webform_openpostcode\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\webform_ui\Form\WebformUiElementEditForm;

/**
 * Hook implementations for webform UI form configuration.
 */
final class FormHooks {

  use StringTranslationTrait;

  /**
   * Helper to add UI and submit mapping for element settings.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  #[Hook('webform_element_configuration_form_alter')]
  public function addElementSettings(array &$form, FormStateInterface $form_state): void {
    /** @var \Drupal\webform_ui\Form\WebformUiElementEditForm $form_object */
    $form_object = $form_state->getFormObject();
    if (!$form_object instanceof WebformUiElementEditForm) {
      return;
    }
    $element_plugin = $form_object->getWebformElementPlugin();
    // Ensure our properties exist on this element.
    if (!$element_plugin->hasProperty('op_enable')) {
      return;
    }

    $group = [
      '#type' => 'details',
      '#title' => $this->t('Open Postcode'),
      '#open' => FALSE,
    ];

    // Build eligible element options from the current webform.
    /** @var \Drupal\webform\WebformInterface $webform */
    $webform = $form_object->getWebform();
    $current_key = (string) $form_object->getKey();
    $elements = is_object($webform) ? $webform->getElementsInitializedAndFlattened() : [];

    $text_options = [];
    $number_options = [];
    foreach ($elements as $key => $el) {
      // Exclude the current element being configured.
      if ($current_key !== '' && $key === $current_key) {
        continue;
      }
      // Skip elements with multiple values, since their input names won't match
      // the simple key (e.g. they use [] suffixes).
      if (!empty($el['#webform_multiple'])) {
        continue;
      }
      $type = $el['#type'] ?? '';
      $title = (string) ($el['#title'] ?? $key);
      $label = $title ? $title . ' (' . $key . ')' : $key;
      // Eligible targets: simple textfields.
      if (in_array($type, ['textfield'], TRUE)) {
        $text_options[$key] = $label;
      }
      // Eligible source (house number): number-like and textfield.
      if (in_array($type, ['number', 'textfield'], TRUE)) {
        $number_options[$key] = $label;
      }
    }

    $group['op_enable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Open Postcode lookup on this element'),
      '#description' => $this->t('Treat this element as the Postcode input and configure related fields.'),
      // Default is auto-populated from the element property; do not set here.
    ];
    $group['op_number_key'] = [
      '#type' => 'select',
      '#title' => $this->t('House number element key'),
      '#description' => $this->t('Webform element key that holds the house number (e.g. "huisnummer").'),
      '#options' => $number_options,
      '#empty_option' => $this->t('- Select -'),
      '#states' => [
        'visible' => [
          ':input[name="properties[op_enable]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $group['op_street_target_key'] = [
      '#type' => 'select',
      '#title' => $this->t('Street target element key'),
      '#description' => $this->t('Webform element key to populate with the street (e.g. "straat").'),
      '#options' => $text_options,
      '#empty_option' => $this->t('- Select -'),
      '#states' => [
        'visible' => [
          ':input[name="properties[op_enable]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $group['op_city_target_key'] = [
      '#type' => 'select',
      '#title' => $this->t('City target element key'),
      '#description' => $this->t('Webform element key to populate with the city (e.g. "woonplaats").'),
      '#options' => $text_options,
      '#empty_option' => $this->t('- Select -'),
      '#states' => [
        'visible' => [
          ':input[name="properties[op_enable]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['open_postcode'] = $group;
  }

}
