<?php

declare(strict_types=1);

namespace Drupal\webform_openpostcode\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\webform_openpostcode\Ajax\PostcodeLookupAjax;

/**
 * Hook implementations for webform element rendering.
 */
final class WebformHooks {

  /**
   * Alter the webform submission form before elements are processed.
   *
   * This hook runs once per form, allowing us to process all elements
   * together before individual element alters are called.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $form_id
   *   The form ID.
   */
  #[Hook('webform_submission_form_alter')]
  public function webformSubmissionFormAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    // First pass: collect all postcode configurations and build element map.
    $postcodeConfigs = [];
    $elementMap = [];
    $this->collectConfigurations($form, $postcodeConfigs, $elementMap);

    // Second pass: apply configurations to all relevant elements.
    $this->applyConfigurations($form, $postcodeConfigs, $elementMap);
  }

  /**
   * Collect postcode lookup configurations and build element map.
   *
   * @param array $form
   *   The form array (passed by reference).
   * @param array $postcodeConfigs
   *   Array to collect configurations (passed by reference).
   * @param array $elementMap
   *   Array to map element keys to element references (passed by reference).
   */
  private function collectConfigurations(array &$form, array &$postcodeConfigs, array &$elementMap): void {
    $this->traverseElements($form, function (array &$element) use (&$postcodeConfigs, &$elementMap): void {
      $webformKey = (string) ($element['#webform_key'] ?? '');
      if ($webformKey === '') {
        return;
      }

      // Store element reference for later use.
      $elementMap[$webformKey] = &$element;

      // If this element has postcode lookup enabled, collect its config.
      if (!empty($element['#op_enable'])) {
        $postcodeConfigs[$webformKey] = [
          'postcode_key' => $webformKey,
          'number_key' => (string) ($element['#op_number_key'] ?? ''),
          'street_target_key' => (string) ($element['#op_street_target_key'] ?? ''),
          'city_target_key' => (string) ($element['#op_city_target_key'] ?? ''),
        ];
      }
    });
  }

  /**
   * Apply postcode lookup configurations to all relevant elements.
   *
   * @param array $form
   *   The form array (passed by reference).
   * @param array $postcodeConfigs
   *   Array of postcode configurations.
   * @param array $elementMap
   *   Array mapping element keys to element references.
   */
  private function applyConfigurations(array &$form, array $postcodeConfigs, array $elementMap): void {
    foreach ($postcodeConfigs as $postcodeKey => $config) {
      // Apply configuration to the postcode element.
      if (isset($elementMap[$postcodeKey])) {
        $this->applyAjaxBehavior($elementMap[$postcodeKey], $config);
      }

      // Also apply to the house number element if specified.
      if ($config['number_key'] !== '' && $config['number_key'] !== $postcodeKey) {
        if (isset($elementMap[$config['number_key']])) {
          $this->applyAjaxBehavior($elementMap[$config['number_key']], $config);
        }
      }
    }
  }

  /**
   * Traverse form elements recursively and apply a callback.
   *
   * @param array $form
   *   The form array (passed by reference).
   * @param callable $callback
   *   Callback to apply to each element with #webform_key.
   */
  private function traverseElements(array &$form, callable $callback): void {
    foreach ($form as $key => &$value) {
      // Skip non-array values.
      if (!is_array($value)) {
        continue;
      }

      // Skip keys that start with '#' as these are element properties, not
      // child elements. In Drupal forms, child elements are always under keys
      // that don't start with '#'.
      if (str_starts_with($key, '#')) {
        continue;
      }

      // If this is a webform element (has #webform_key), process it.
      if (isset($value['#webform_key'])) {
        $callback($value);
      }

      // Recursively process nested elements.
      $this->traverseElements($value, $callback);
    }
  }

  /**
   * Attach Ajax metadata to an element if not already applied.
   *
   * @param array $element
   *   The element render array.
   * @param array $config
   *   The lookup configuration.
   */
  private function applyAjaxBehavior(array &$element, array $config): void {
    if (isset($element['#webform_openpostcode']) && $element['#webform_openpostcode'] === $config) {
      return;
    }

    $element['#webform_openpostcode'] = $config;
    $element['#ajax'] = [
      'callback' => [PostcodeLookupAjax::class, 'handle'],
      'event' => 'change',
      'progress' => [
        'type' => 'throbber',
      ],
    ];
  }

}
