<?php

declare(strict_types=1);

namespace Drupal\Tests\webform_openpostcode\Unit\Ajax;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\webform_openpostcode\Ajax\PostcodeLookupAjax;
use Drupal\webform_openpostcode\Service\OpenPostcodeLookupInterface;
use Prophecy\PhpUnit\ProphecyTrait;

/**
 * @covers \Drupal\webform_openpostcode\Ajax\PostcodeLookupAjax
 *
 * @group webform_openpostcode
 */
final class PostcodeLookupAjaxTest extends UnitTestCase {
  use ProphecyTrait;

  /**
   * Set up container with a stubbed lookup service.
   */
  protected function setUp(): void {
    parent::setUp();
    // Provide a container with our stubbed lookup service.
    $container = new ContainerBuilder();
    $stub = new class() implements OpenPostcodeLookupInterface {
      /**
       * @var array<int, array>
       */
      public array $calls = [];

      /**
       * {@inheritdoc}
       */
      public function lookup(string $postcode, string $houseNumber): array {
        $this->calls[] = [$postcode, $houseNumber];
        return ['straat' => 'Damrak', 'woonplaats' => 'Amsterdam'];
      }

    };
    $container->set(OpenPostcodeLookupInterface::class, $stub);
    \Drupal::setContainer($container);
  }

  /**
   * Ensures handler returns no commands when config/values are missing.
   */
  public function testEarlyReturnWhenConfigMissingOrEmptyValues(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->method('getTriggeringElement')->willReturn(['#name' => 'noop']);
    $form_state->method('getValues')->willReturn([]);

    $form = [];
    $response = PostcodeLookupAjax::handle($form, $form_state);
    $this->assertSame([], $response->getCommands());
  }

  /**
   * Ensures targets are populated only if currently empty.
   */
  public function testPopulatesTargetsWhenEmpty(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $trigger = [
      '#webform_openpostcode' => [
        'postcode_key' => 'postcode',
        'number_key' => 'huisnummer',
        'street_target_key' => 'straat',
        'city_target_key' => 'woonplaats',
      ],
    ];

    $form_state->method('getTriggeringElement')->willReturn($trigger);
    $form_state->method('getValues')->willReturn([
      'postcode' => '1012AB',
      'huisnummer' => '1',
      'straat' => '',
      'woonplaats' => '',
    ]);

    $form = [];
    $response = PostcodeLookupAjax::handle($form, $form_state);
    $commands = $response->getCommands();
    // Expect three InvokeCommand groups:
    // 1. set value + trigger change for each target.
    // 2. set value + trigger change for postcode (formatting)
    $this->assertNotEmpty($commands);

    // Extract only the 'invoke' commands.
    $invoke = array_values(array_filter(
      $commands,
      static fn ($c) => ($c['command'] ?? '') === 'invoke'
    ));
    $this->assertCount(6, $invoke);
    $this->assertSame(':input[name="straat"]', $invoke[0]['selector']);
    $this->assertSame('val', $invoke[0]['method']);
    $this->assertSame(['Damrak'], $invoke[0]['args']);
    $this->assertSame(':input[name="straat"]', $invoke[1]['selector']);
    $this->assertSame('trigger', $invoke[1]['method']);
    $this->assertSame(['change'], $invoke[1]['args']);

    $this->assertSame(':input[name="woonplaats"]', $invoke[2]['selector']);
    $this->assertSame('val', $invoke[2]['method']);
    $this->assertSame(['Amsterdam'], $invoke[2]['args']);
    $this->assertSame(':input[name="woonplaats"]', $invoke[3]['selector']);
    $this->assertSame('trigger', $invoke[3]['method']);
    $this->assertSame(['change'], $invoke[3]['args']);
  }

  /**
   * Ensures existing target values are not overridden by Ajax handler.
   */
  public function testDoesNotOverrideWhenTargetsAlreadyHaveValues(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $trigger = [
      '#webform_openpostcode' => [
        'postcode_key' => 'postcode',
        'number_key' => 'huisnummer',
        'street_target_key' => 'straat',
        'city_target_key' => 'woonplaats',
      ],
    ];
    $form_state->method('getTriggeringElement')->willReturn($trigger);
    $form_state->method('getValues')->willReturn([
      'postcode' => '1012AB',
      'huisnummer' => '1',
      'straat' => 'Existing street',
      'woonplaats' => 'Existing city',
    ]);

    $form = [];
    $response = PostcodeLookupAjax::handle($form, $form_state);
    // We still expect 2 commands for changing the postcode (formatting).
    $this->assertCount(2, $response->getCommands());
  }

}
