<?php

declare(strict_types=1);

namespace Drupal\Tests\webform_openpostcode\Unit\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\webform_openpostcode\Ajax\PostcodeLookupAjax;
use Drupal\webform_openpostcode\Hook\WebformHooks;
use Prophecy\PhpUnit\ProphecyTrait;

/**
 * @covers \Drupal\webform_openpostcode\Hook\WebformHooks
 * @group webform_openpostcode
 */
final class WebformHooksTest extends UnitTestCase {
  use ProphecyTrait;

  /**
   * Test that elements without op_enable are not modified.
   */
  public function testNoBehaviorWhenNotEnabled(): void {
    $hooks = new WebformHooks();
    $form_state = $this->prophesize(FormStateInterface::class)->reveal();

    $form = [
      'postcode' => [
        '#webform_key' => 'postcode',
        '#type' => 'textfield',
        // '#op_enable' is absent or falsy.
      ],
    ];

    $hooks->webformSubmissionFormAlter($form, $form_state, 'test_form');

    $this->assertArrayNotHasKey('#webform_openpostcode', $form['postcode']);
    $this->assertArrayNotHasKey('#ajax', $form['postcode']);
  }

  /**
   * Test that Ajax behavior is added when op_enable is set.
   */
  public function testAjaxBehaviorAddedWhenEnabled(): void {
    $hooks = new WebformHooks();
    $form_state = $this->prophesize(FormStateInterface::class)->reveal();

    $form = [
      'postcode' => [
        '#webform_key' => 'postcode',
        '#type' => 'textfield',
        '#op_enable' => 1,
        '#op_number_key' => 'huisnummer',
        '#op_street_target_key' => 'straat',
        '#op_city_target_key' => 'woonplaats',
      ],
    ];

    $hooks->webformSubmissionFormAlter($form, $form_state, 'test_form');

    $this->assertSame([
      'postcode_key' => 'postcode',
      'number_key' => 'huisnummer',
      'street_target_key' => 'straat',
      'city_target_key' => 'woonplaats',
    ], $form['postcode']['#webform_openpostcode']);

    $this->assertArrayHasKey('#ajax', $form['postcode']);
    $this->assertSame([PostcodeLookupAjax::class, 'handle'], $form['postcode']['#ajax']['callback']);
    $this->assertSame('change', $form['postcode']['#ajax']['event']);
    $this->assertSame('throbber', $form['postcode']['#ajax']['progress']['type']);
  }

  /**
   * Test that referenced number element also gets Ajax behavior.
   */
  public function testNumberElementGetsAjaxBehavior(): void {
    $hooks = new WebformHooks();
    $form_state = $this->prophesize(FormStateInterface::class)->reveal();

    $form = [
      'postcode' => [
        '#webform_key' => 'postcode',
        '#type' => 'textfield',
        '#op_enable' => 1,
        '#op_number_key' => 'huisnummer',
        '#op_street_target_key' => 'straat',
        '#op_city_target_key' => 'woonplaats',
      ],
      'huisnummer' => [
        '#webform_key' => 'huisnummer',
        '#type' => 'number',
      ],
    ];

    $hooks->webformSubmissionFormAlter($form, $form_state, 'test_form');

    // Postcode element should have Ajax behavior.
    $this->assertArrayHasKey('#ajax', $form['postcode']);
    $this->assertSame('postcode', $form['postcode']['#webform_openpostcode']['postcode_key']);
    $this->assertSame('huisnummer', $form['postcode']['#webform_openpostcode']['number_key']);

    // House number element should also have Ajax behavior.
    $this->assertArrayHasKey('#ajax', $form['huisnummer']);
    $this->assertSame('postcode', $form['huisnummer']['#webform_openpostcode']['postcode_key']);
    $this->assertSame('huisnummer', $form['huisnummer']['#webform_openpostcode']['number_key']);
  }

  /**
   * Test that nested form elements are processed correctly.
   */
  public function testNestedFormElements(): void {
    $hooks = new WebformHooks();
    $form_state = $this->prophesize(FormStateInterface::class)->reveal();

    $form = [
      'container' => [
        '#type' => 'container',
        'postcode' => [
          '#webform_key' => 'postcode',
          '#type' => 'textfield',
          '#op_enable' => 1,
          '#op_number_key' => 'huisnummer',
        ],
        'huisnummer' => [
          '#webform_key' => 'huisnummer',
          '#type' => 'number',
        ],
      ],
    ];

    $hooks->webformSubmissionFormAlter($form, $form_state, 'test_form');

    $this->assertArrayHasKey('#ajax', $form['container']['postcode']);
    $this->assertArrayHasKey('#ajax', $form['container']['huisnummer']);
  }

  /**
   * Test that elements without webform_key are skipped.
   */
  public function testElementsWithoutWebformKeyAreSkipped(): void {
    $hooks = new WebformHooks();
    $form_state = $this->prophesize(FormStateInterface::class)->reveal();

    $form = [
      'regular_field' => [
        '#type' => 'textfield',
        // No #webform_key.
      ],
      'postcode' => [
        '#webform_key' => 'postcode',
        '#type' => 'textfield',
        '#op_enable' => 1,
      ],
    ];

    $hooks->webformSubmissionFormAlter($form, $form_state, 'test_form');

    $this->assertArrayNotHasKey('#webform_openpostcode', $form['regular_field']);
    $this->assertArrayHasKey('#webform_openpostcode', $form['postcode']);
  }

  /**
   * Test that multiple postcode configurations work independently.
   */
  public function testMultiplePostcodeConfigurations(): void {
    $hooks = new WebformHooks();
    $form_state = $this->prophesize(FormStateInterface::class)->reveal();

    $form = [
      'postcode1' => [
        '#webform_key' => 'postcode1',
        '#type' => 'textfield',
        '#op_enable' => 1,
        '#op_number_key' => 'number1',
      ],
      'number1' => [
        '#webform_key' => 'number1',
        '#type' => 'number',
      ],
      'postcode2' => [
        '#webform_key' => 'postcode2',
        '#type' => 'textfield',
        '#op_enable' => 1,
        '#op_number_key' => 'number2',
      ],
      'number2' => [
        '#webform_key' => 'number2',
        '#type' => 'number',
      ],
    ];

    $hooks->webformSubmissionFormAlter($form, $form_state, 'test_form');

    $this->assertArrayHasKey('#ajax', $form['postcode1']);
    $this->assertArrayHasKey('#ajax', $form['number1']);
    $this->assertSame('postcode1', $form['number1']['#webform_openpostcode']['postcode_key']);

    $this->assertArrayHasKey('#ajax', $form['postcode2']);
    $this->assertArrayHasKey('#ajax', $form['number2']);
    $this->assertSame('postcode2', $form['number2']['#webform_openpostcode']['postcode_key']);
  }

  /**
   * Test that behavior is not applied twice to the same element.
   */
  public function testBehaviorNotAppliedTwice(): void {
    $hooks = new WebformHooks();
    $form_state = $this->prophesize(FormStateInterface::class)->reveal();

    $form = [
      'postcode' => [
        '#webform_key' => 'postcode',
        '#type' => 'textfield',
        '#op_enable' => 1,
        '#op_number_key' => 'postcode',
        // Same element is both postcode and number.
      ],
    ];

    $hooks->webformSubmissionFormAlter($form, $form_state, 'test_form');

    // Should have Ajax configuration applied only once.
    $this->assertArrayHasKey('#ajax', $form['postcode']);
    $this->assertArrayHasKey('#webform_openpostcode', $form['postcode']);
    $this->assertSame('postcode', $form['postcode']['#webform_openpostcode']['postcode_key']);
  }

  /**
   * Test that empty string number_key is handled correctly.
   */
  public function testEmptyNumberKey(): void {
    $hooks = new WebformHooks();
    $form_state = $this->prophesize(FormStateInterface::class)->reveal();

    $form = [
      'postcode' => [
        '#webform_key' => 'postcode',
        '#type' => 'textfield',
        '#op_enable' => 1,
        '#op_number_key' => '',
        '#op_street_target_key' => 'straat',
      ],
      'huisnummer' => [
        '#webform_key' => 'huisnummer',
        '#type' => 'number',
      ],
    ];

    $hooks->webformSubmissionFormAlter($form, $form_state, 'test_form');

    // Postcode should have Ajax behavior.
    $this->assertArrayHasKey('#ajax', $form['postcode']);

    // House number should NOT have Ajax behavior when number_key is empty.
    $this->assertArrayNotHasKey('#ajax', $form['huisnummer']);
  }

}
