<?php

declare(strict_types=1);

namespace Drupal\Tests\webform_openpostcode\Unit\Service;

use Drupal\Core\Cache\CacheBackendInterface;

/**
 * Simple in-memory cache backend for testing.
 */
final class InMemoryCacheBackend implements CacheBackendInterface {

  /**
   * Array to keep the cache data.
   *
   * @var array<string, object>
   */
  private array $store = [];

  /**
   * {@inheritdoc}
   */
  public function get($cid, $allow_invalid = FALSE): ?object {
    return $this->store[$cid] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getMultiple(&$cids, $allow_invalid = FALSE): array {
    $results = [];
    foreach ($cids as $cid) {
      if (isset($this->store[$cid])) {
        $results[$cid] = $this->store[$cid];
      }
    }
    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function set($cid, $data, $expire = CacheBackendInterface::CACHE_PERMANENT, array $tags = []): void {
    $this->store[$cid] = (object) [
      'cid' => $cid,
      'data' => $data,
      'expire' => $expire,
      'tags' => $tags,
      'valid' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function setMultiple(array $items): void {
    foreach ($items as $cid => $item) {
      $this->set($cid, $item['data'], $item['expire'] ?? CacheBackendInterface::CACHE_PERMANENT, $item['tags'] ?? []);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function delete($cid): void {
    unset($this->store[$cid]);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteMultiple(array $cids): void {
    foreach ($cids as $cid) {
      $this->delete($cid);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAll(): void {
    $this->store = [];
  }

  /**
   * {@inheritdoc}
   */
  public function invalidate($cid): void {
    // No-op for unit test.
  }

  /**
   * {@inheritdoc}
   */
  public function invalidateMultiple(array $cids): void {
    // No-op for unit test.
  }

  /**
   * {@inheritdoc}
   */
  public function invalidateAll(): void {
    // No-op for unit test.
  }

  /**
   * {@inheritdoc}
   */
  public function garbageCollection(): void {
    // No-op for unit test.
  }

  /**
   * {@inheritdoc}
   */
  public function removeBin(): void {
    $this->store = [];
  }

}
