<?php

declare(strict_types=1);

namespace Drupal\webform_openpostcode\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use GuzzleHttp\ClientInterface;

/**
 * Default Open Postcode lookup implementation.
 */
final class OpenPostcodeLookup implements OpenPostcodeLookupInterface {

  public function __construct(
    private readonly ClientInterface $httpClient,
    private readonly CacheBackendInterface $cache,
    private readonly LoggerChannelInterface $logger,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function lookup(string $postcode, string $houseNumber): array {
    $postcode = strtoupper(str_replace(' ', '', trim($postcode)));
    $houseNumber = trim($houseNumber);

    if ($postcode === '' || $houseNumber === '') {
      throw new \InvalidArgumentException('Missing postcode or house number.');
    }

    $cid = sprintf('webform_openpostcode:%s:%s', $postcode, $houseNumber);
    if ($cache = $this->cache->get($cid)) {
      return (array) $cache->data;
    }

    try {
      $response = $this->httpClient->request('GET', 'https://openpostcode.nl/api/address', [
        'query' => [
          'postcode' => $postcode,
          'huisnummer' => $houseNumber,
        ],
        'timeout' => 3.0,
        'connect_timeout' => 2.0,
        'allow_redirects' => FALSE,
      ]);
      $decoded = json_decode((string) $response->getBody(), TRUE) ?? [];
    }
    catch (\Throwable $throwable) {
      $this->logger->error('Open Postcode lookup failed: @message', [
        '@message' => $throwable->getMessage(),
      ]);
      throw new \RuntimeException('Lookup failed.', 0, $throwable);
    }

    $data = [
      'straat' => (string) ($decoded['straat'] ?? ''),
      'woonplaats' => (string) ($decoded['woonplaats'] ?? ''),
    ];

    $this->cache->set($cid, $data, time() + 60 * 60 * 12);
    return $data;
  }

}
