<?php

declare(strict_types=1);

namespace Drupal\Tests\webform_openpostcode\Unit\Service;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\webform_openpostcode\Service\OpenPostcodeLookup;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Psr7\Response;

/**
 * @covers \Drupal\webform_openpostcode\Service\OpenPostcodeLookup
 * @group webform_openpostcode
 */
final class OpenPostcodeLookupTest extends UnitTestCase {

  /**
   * Tests that lookup throws an exception when input is missing.
   */
  public function testThrowsOnMissingInput(): void {
    $service = new OpenPostcodeLookup(
      $this->createMock(ClientInterface::class),
      new InMemoryCacheBackend(),
      $this->createMock(LoggerChannelInterface::class),
    );

    $this->expectException(\InvalidArgumentException::class);
    $service->lookup('', '10');
  }

  /**
   * Tests that a successful lookup caches the response and maps it correctly.
   */
  public function testSuccessfulLookupCachesAndMapsResponse(): void {
    $httpClient = $this->createMock(ClientInterface::class);
    $logger = $this->createMock(LoggerChannelInterface::class);
    $cache = new InMemoryCacheBackend();

    $responseData = [
      'straat' => 'Kalverstraat',
      'woonplaats' => 'Amsterdam',
    ];
    $httpClient->expects($this->once())
      ->method('request')
      ->with(
        'GET',
        'https://openpostcode.nl/api/address',
        $this->callback(function (array $options): bool {
          $this->assertArrayHasKey('query', $options);
          $this->assertSame('1012AB', $options['query']['postcode']);
          $this->assertSame('12', $options['query']['huisnummer']);
          return TRUE;
        })
      )
      ->willReturn(new Response(200, [], json_encode($responseData)));

    $service = new OpenPostcodeLookup($httpClient, $cache, $logger);

    // First call hits HTTP and caches.
    $result1 = $service->lookup('1012 ab', '12');
    $this->assertSame($responseData, $result1);

    // Second call is served from cache (HTTP still expected once only).
    $result2 = $service->lookup('1012AB', '12');
    $this->assertSame($responseData, $result2);
  }

  /**
   * Tests that HTTP failures are logged and throw a runtime exception.
   */
  public function testHttpFailureIsLoggedAndThrowsRuntime(): void {
    $httpClient = $this->createMock(ClientInterface::class);
    $cache = new InMemoryCacheBackend();
    $logger = $this->createMock(LoggerChannelInterface::class);
    $logger->expects($this->once())->method('error');

    $httpClient->expects($this->once())
      ->method('request')
      ->willThrowException(new \RuntimeException('network error'));

    $service = new OpenPostcodeLookup($httpClient, $cache, $logger);

    $this->expectException(\RuntimeException::class);
    $service->lookup('1012AB', '12');
  }

}
