<?php

namespace Drupal\webform_payment_element\Element;

use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element\CompositeFormElementTrait;
use Drupal\Core\Render\Element\FormElementBase;
use Drupal\webform\Utility\WebformElementHelper;
use Drupal\webform\Utility\WebformOptionsHelper;

/**
 * Provides a Webform Payment Element form element.
 *
 * @FormElement("webform_payment_element")
 *
 * @see \Drupal\Core\Render\Element\FormElementBase
 * @see https://api.drupal.org/api/drupal/core%21lib%21Drupal%21Core%21Render%21Element%21FormElementBase.php/class/FormElementBase
 * @see \Drupal\Core\Render\Element\RenderElementBase
 * @see https://api.drupal.org/api/drupal/namespace/Drupal%21Core%21Render%21Element
 * @see \Drupal\webform_example_element\Element\WebformExampleElement
 */
class WebformPaymentElement extends FormElementBase {

  use CompositeFormElementTrait;
  const CHECKBOXRADIO = 'checkboxradio';
  const SELECT = 'select';
  const TABLESELECT = 'tableselect';

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    return [
      '#input' => TRUE,
      '#mode' => static::TABLESELECT,
      '#items' => [],
      '#amount_precision' => 2,
      '#amount_prefix' => '$ ',
      '#amount_suffix' => '',
      '#amount_thousands_separator' => ',',
      '#amount_decimal_separator' => '.',
      '#multiple' => FALSE,
      '#required' => FALSE,
      '#process' => [
        [static::class, 'processWebformPaymentElement'],
      ],
      '#theme_wrappers' => ['form_element'],
    ];
  }

  /**
   * Processes a 'webform_payment_element' element.
   */
  public static function processWebformPaymentElement(array &$element, FormStateInterface $form_state, array &$complete_form): array {
    $element['#tree'] = TRUE;

    // Define the default value.
    if (!isset($element['#default_value'])) {
      $element['#default_value'] = !empty($element['#multiple']) ? [] : NULL;
    }

    $items = $element['#items'];

    // Prepare items for display via tableselect, checkboxradio, or select.
    foreach ($items as $key => $item) {
      $items[$key]['amount'] = static::formatPaymentAmount($element, $item['amount']);
    }

    // Build checkboxes/radios element or select.
    if (in_array($element['#mode'], [static::CHECKBOXRADIO, static::SELECT])) {
      $options = [];
      foreach ($items as $key => $item) {
        $options[$key] = $item['label'] . ' - ' . $item['amount'];
        if ($element['#mode'] === static::CHECKBOXRADIO && isset($item['description'])) {
          $options[$key] .= WebformOptionsHelper::DESCRIPTION_DELIMITER . $item['description'];
        }
      }

      $item_element_type = ($element['#mode'] === static::CHECKBOXRADIO)
        ? ($element['#multiple'] ? 'checkboxes' : 'radios')
        : 'select';

      $element['items'] = [
        '#type' => $item_element_type,
        '#options' => $options,
        '#multiple' => $element['#multiple'],
        '#required' => $element['#required'],
        '#default_value' => $element['#default_value'],
      ];

      // Enhance checkbox/radio element description using ( -- ) separator.
      $items_element =& $element['items'];
      WebformElementHelper::process($items_element);
    }

    // Default is table select.
    elseif ($element['#mode'] === static::TABLESELECT) {
      $has_description = FALSE;
      foreach ($items as $key => $item) {
        $has_description = $has_description ?: isset($item['description']);
        $items[$key]['label'] = [
          'data' => [
            '#markup' => (isset($item['description']))
              ? '<strong>' . $item['label'] . '</strong><br/>' . $item['description']
              : '<strong>' . $item['label'] . '</strong>',
          ],
        ];
        unset($item['description']);
      }

      $header = [];
      $header['label'] = $has_description
        ? t('Label / Description')
        : t('Label');
      $header['amount'] = t('Amount');

      // Massage the default value for multiple values for table select.
      if (!empty($element['#multiple']) && !empty($element['#default_value'])) {
        $element['#default_value'] = array_combine($element['#default_value'], $element['#default_value']);
      }

      $element['items'] = [
        '#type' => 'tableselect',
        '#empty' => t('No items available.'),
        '#header' => $header,
        '#options' => $items,
        '#multiple' => $element['#multiple'],
        '#required' => $element['#required'],
        '#default_value' => $element['#default_value'] ?? NULL,
      ];
    }
    else {
      throw new \InvalidArgumentException(sprintf('Invalid payment element mode: %s', $element['#mode']));
    }

    // Add validate callback.
    $element += ['#element_validate' => []];
    array_unshift($element['#element_validate'], [get_called_class(), 'validateWebformPaymentElement']);

    return $element;
  }

  /**
   * Validates a 'webform_payment_element' element.
   */
  public static function validateWebformPaymentElement(array &$element, FormStateInterface $form_state, array &$complete_form): void {
    $value = NestedArray::getValue($form_state->getValues(), $element['#parents']);
    if ($element['#multiple']) {
      $form_state->setValueForElement($element, array_values(array_filter($value['items'])));
    }
    else {
      $form_state->setValueForElement($element, $value['items']);
    }
  }

  /**
   * Formats a payment item into a displayable string.
   *
   * @param array $element
   *   The webform payment element containing #items.
   * @param string $item
   *   The key of the payment item to format.
   *
   * @return string
   *   The formatted payment item string consisting of the label and amount,
   *   or the item key if the amount is not available.
   */
  public static function formatPaymentItem(array $element, string $item): string {
    $amount = NestedArray::getValue($element, ['#items', $item, 'amount']);
    if (is_null($amount)) {
      return $item;
    }

    $amount = static::formatPaymentAmount($element, $amount);
    $label = NestedArray::getValue($element, ['#items', $item, 'label']);
    return ($label)
      ? $label . ' - ' . $amount
      : $amount;
  }

  /**
   * Formats the payment amount based on the given element's configuration.
   *
   * @param array $element
   *   The webform payment element containing formatting configuration, such as
   *   precision, decimal separator, and thousands separator.
   * @param mixed $amount
   *   The payment amount to be formatted.
   *
   * @return string
   *   The formatted payment amount as a string.
   */
  public static function formatPaymentAmount(array $element, mixed $amount): string {
    $element += [
      '#amount_precision' => 2,
      '#amount_prefix' => '$ ',
      '#amount_suffix' => '',
      '#amount_thousands_separator' => ',',
      '#amount_decimal_separator' => '.',
    ];

    // Format the amount.
    $amount = number_format(
      (float) $amount,
      $element['#amount_precision'],
      $element['#amount_decimal_separator'],
      $element['#amount_thousands_separator']
    );

    // Wrap the amount in the amount prefix and suffix.
    return $element['#amount_prefix'] . $amount . $element['#amount_suffix'];
  }

  /**
   * Calculates the total payment amount based on the specified values.
   *
   * @param array $element
   *   The webform payment element with #items.
   * @param array|string|null $values
   *   A single value or an array of values containing the selected payment items.
   *
   * @return float
   *   The total payment amount.
   */
  public static function getPaymentTotal(array $element, array|string|null $values): float {
    if (is_null($values)) {
      return 0.0;
    }

    $total = 0.0;
    $values = (array) $values;
    foreach ($values as $value) {
      $total += (float) NestedArray::getValue($element['#items'], [$value, 'amount']) ?? 0.0;
    }
    return $total;
  }

}
