<?php

namespace Drupal\webform_payment_element\Element;

use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\Element\WebformComputedBase;
use Drupal\webform\WebformSubmissionForm;
use Drupal\webform\WebformSubmissionInterface;

/**
 * Provides an item to display the payment total for a webform submission.
 *
 * @RenderElement("webform_payment_element_total")
 */
class WebformPaymentElementTotal extends WebformComputedBase {

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    return parent::getInfo() + [
      '#amount_precision' => 2,
      '#amount_prefix' => '$ ',
      '#amount_suffix' => '',
      '#amount_thousands_separator' => ',',
      '#amount_decimal_separator' => '.',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function getMode(array $element) {
    return 'text';
  }

  /**
   * {@inheritdoc}
   */
  public static function processWebformComputed(&$element, FormStateInterface $form_state, &$complete_form) {
    // Append payment element keys to the #template to trigger an Ajax refresh.
    // @see \Drupal\webform\Element\WebformComputedBase::processWebformComputed
    $form_object = $form_state->getFormObject();
    if ($element['#ajax'] && $form_object instanceof WebformSubmissionForm) {
      $webform = $form_object->getWebform();
      $webform_elements = $webform->getElementsDecodedAndFlattened();
      $element['#template'] = '';
      foreach ($webform_elements as $webform_element_key => $webform_element) {
        if (isset($webform_element['#type'])
          && $webform_element['#type'] === 'webform_payment_element') {
          $element['#template'] .= 'data.' . $webform_element_key . PHP_EOL;
        }
      }
    }

    return parent::processWebformComputed($element, $form_state, $complete_form);
  }

  /**
   * {@inheritdoc}
   */
  public static function computeValue(array $element, WebformSubmissionInterface $webform_submission) {
    $total = static::getPaymentTotal($webform_submission);
    return WebformPaymentElement::formatAmount($element, $total);
  }

  /**
   * Calculates the total payment amount from a webform submission.
   *
   * @param \Drupal\webform\WebformSubmissionInterface $webform_submission
   *   The webform submission.
   *
   * @return float
   *   The total payment amount derived from the payment elements in the webform submission.
   */
  public static function getPaymentTotal(WebformSubmissionInterface $webform_submission): float {
    $total = 0;
    $elements = $webform_submission->getWebform()->getElementsDecodedAndFlattened();
    foreach ($elements as $key => $element) {
      $type = $element['#type'] ?? NULL;
      if ($type === 'webform_payment_element') {
        $total += WebformPaymentElement::getElementTotal($element, $webform_submission->getElementData($key));
      }
    }
    return $total;
  }

}
