<?php

namespace Drupal\webform_payment_element\Plugin\WebformElement;

use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\Plugin\WebformElementBase;
use Drupal\webform\WebformInterface;
use Drupal\webform\WebformSubmissionInterface;
use Drupal\webform_payment_element\Element\WebformPaymentElement as WebformPaymentElementElement;

/**
 * Provides a 'webform_payment_element' element.
 *
 * @WebformElement(
 *   id = "webform_payment_element",
 *   label = @Translation("Webform payment element"),
 *   description = @Translation("Provides a webform element payment."),
 *   category = @Translation("Payment elements"),
 * )
 *
 * @see \Drupal\webform\Plugin\WebformElement\OptionsBase
 */
class WebformPaymentElement extends WebformElementBase {

  /**
   * {@inheritdoc}
   */
  protected function defineDefaultProperties() {
    return [
      'multiple' => FALSE,
      'mode' => WebformPaymentElementElement::TABLESELECT,
      'items' => [],
      'amount_precision' => 2,
      'amount_prefix' => '$ ',
      'amount_suffix' => '',
      'amount_thousands_separator' => ',',
      'amount_decimal_separator' => '.',
    ] + parent::defineDefaultProperties();
  }

  /* ************************************************************************ */

  /**
   * {@inheritdoc}
   */
  public function supportsMultipleValues() {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function hasMultipleWrapper() {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    $form['webform_payment'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Payment'),
    ];

    $form['webform_payment']['mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Payment mode'),
      '#required' => TRUE,
      '#options' => [
        WebformPaymentElementElement::TABLESELECT => $this->t('Table select'),
        WebformPaymentElementElement::CHECKBOXRADIO => $this->t('Checkboxes/Radios'),
        WebformPaymentElementElement::SELECT => $this->t('Select list'),
      ],
    ];

    // @see \Drupal\webform\Element\WebformMultiple
    $form['webform_payment']['items'] = [
      '#type' => 'webform_multiple',
      '#title' => $this->t('Payment items'),
      '#description' => $this->t('Enter payment items.'),
      '#key' => 'name',
      '#add' => FALSE,
      '#header' => [
        'name' => ['data' => $this->t('Name'), 'width' => '40%'],
        'item_text' => ['data' => $this->t('Amount / Label / Description'), 'width' => '60%'],
      ],
      '#required' => TRUE,
      '#min_items' => 1,
      '#element' => [
        'name' => [
          '#type' => 'textfield',
          '#title' => $this->t('Name'),
          '#title_display' => 'invisible',
          '#placeholder' => $this->t('Enter name…'),
          '#required' => TRUE,
        ],
        'item_text' => [
          'amount' => [
            '#type' => 'number',
            '#title' => $this->t('Amount'),
            '#title_display' => 'invisible',
            '#placeholder' => $this->t('Enter amount…'),
            '#required' => TRUE,
          ],
          'label' => [
            '#type' => 'textfield',
            '#title' => $this->t('Label'),
            '#title_display' => 'invisible',
            '#placeholder' => $this->t('Enter label…'),
            '#required' => TRUE,
          ],
          'description' => [
            '#type' => 'textarea',
            '#title' => $this->t('Description'),
            '#title_display' => 'invisible',
            '#placeholder' => $this->t('Enter description…'),
            '#rows' => 2,
            '#states' => [
              '!visible' => [
                ':input[name="properties[mode]"]' => ['value' => 'select'],
              ],
            ],

          ],
        ],
      ],
    ];

    $form['webform_payment']['amount'] = [
      '#type' => 'details',
      '#title' => $this->t('Payment amount'),
      '#description' => $this->t('Enter how you want the payment amount to be displayed.'),
    ];
    $form['webform_payment']['amount']['amount_precision'] = [
      '#type' => 'select',
      '#title' => $this->t('Precision'),
      '#description' => $this->t('The number of decimal places to display in the amount (0-10).'),
      '#required' => TRUE,
      '#options' => range(0, 5),
    ];
    $form['webform_payment']['amount']['parts'] = $this->getFormInlineContainer();
    $form['webform_payment']['amount']['parts']['amount_prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Prefix'),
      '#description' => $this->t('Text or symbol to display before the amount (e.g., "$").'),
    ];
    $form['webform_payment']['amount']['parts']['amount_suffix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Suffix'),
      '#description' => $this->t('Text or symbol to display after the amount (e.g., "USD").'),
    ];

    $form['webform_payment']['amount']['separator'] = $this->getFormInlineContainer();
    $form['webform_payment']['amount']['separator']['amount_thousands_separator'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Thousands separator'),
      '#description' => $this->t('Character to use as thousands separator (e.g., ",").'),
    ];
    $form['webform_payment']['amount']['separator']['amount_decimal_separator'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Decimal separator'),
      '#description' => $this->t('Character to use as decimal separator (e.g., ".").'),
    ];
    return $form;
  }

  /**
   * Format an element's value as text.
   *
   * @param array $element
   *   An element.
   * @param \Drupal\webform\WebformSubmissionInterface $webform_submission
   *   A webform submission.
   * @param array $options
   *   An array of options.
   *
   * @return string
   *   The element's value formatted as text.
   *
   * @see \Drupal\webform\Hook\WebformTokensHooks::getSubmissionValue()
   */
  protected function formatTextItem(array $element, WebformSubmissionInterface $webform_submission, array $options = []): string {
    $value = $this->getValue($element, $webform_submission, $options);
    $format = $this->getItemFormat($element);

    if ($format === 'raw') {
      return $value;
    }

    return WebformPaymentElementElement::formatPaymentItem($element, $value);
  }

  /**
   * {@inheritdoc}
   */
  public function getTestValues(array $element, WebformInterface $webform, array $options = []) {
    if (empty($element['#items'])) {
      return NULL;
    }

    return array_rand(
      $element['#items'],
      !empty($element['#multiple']) ? 2 : 1,
    );
  }

}
