<?php

namespace Drupal\Tests\webform_payment_element\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\webform_payment_element\Element\WebformPaymentElement;

/**
 * Tests for WebformPaymentElement methods.
 *
 * @coversDefaultClass \Drupal\webform_payment_element\Element\WebformPaymentElement
 */
class WebformPaymentElementUnitTest extends UnitTestCase {

  /**
   * Tests formatAmount method.
   *
   * @covers ::formatAmount
   * @dataProvider providerFormatAmount
   */
  public function testFormatAmount(array $element, mixed $amount, string $expected): void {
    $result = WebformPaymentElement::formatAmount($element, $amount);
    $this->assertEquals($expected, $result);
  }

  /**
   * Data provider for testFormatAmount.
   *
   * @return array
   *   Test cases with element config, amount, and expected output.
   */
  public static function providerFormatAmount(): array {
    return [
      'default formatting' => [
        'element' => [],
        'amount' => 100,
        'expected' => '$ 100.00',
      ],
      'custom prefix and suffix' => [
        'element' => [
          '#amount_prefix' => 'USD ',
          '#amount_suffix' => ' (tax incl.)',
        ],
        'amount' => 50.99,
        'expected' => 'USD 50.99 (tax incl.)',
      ],
      'no prefix or suffix' => [
        'element' => [
          '#amount_prefix' => '',
          '#amount_suffix' => '',
        ],
        'amount' => 25.50,
        'expected' => '25.50',
      ],
      'custom precision' => [
        'element' => [
          '#amount_precision' => 0,
        ],
        'amount' => 99.99,
        'expected' => '$ 100',
      ],
      'three decimal places' => [
        'element' => [
          '#amount_precision' => 3,
        ],
        'amount' => 123.456,
        'expected' => '$ 123.456',
      ],
      'custom separators' => [
        'element' => [
          '#amount_prefix' => '€ ',
          '#amount_thousands_separator' => '.',
          '#amount_decimal_separator' => ',',
        ],
        'amount' => 1234.56,
        'expected' => '€ 1.234,56',
      ],
      'large amount with thousands separator' => [
        'element' => [],
        'amount' => 1000000.99,
        'expected' => '$ 1,000,000.99',
      ],
      'zero amount' => [
        'element' => [],
        'amount' => 0,
        'expected' => '$ 0.00',
      ],
      'string amount conversion' => [
        'element' => [],
        'amount' => '42.50',
        'expected' => '$ 42.50',
      ],
    ];
  }

  /**
   * Tests formatPaymentItem method.
   *
   * @covers ::formatPaymentItem
   * @dataProvider providerFormatPaymentItem
   */
  public function testFormatPaymentItem(array $element, string $item, string $expected): void {
    $result = WebformPaymentElement::formatPaymentItem($element, $item);
    $this->assertEquals($expected, $result);
  }

  /**
   * Data provider for testFormatPaymentItem.
   *
   * @return array
   *   Test cases with element config, item key, and expected output.
   */
  public static function providerFormatPaymentItem(): array {
    return [
      'item with label and amount' => [
        'element' => [
          '#items' => [
            'basic' => [
              'label' => 'Basic Plan',
              'amount' => 10.00,
            ],
          ],
        ],
        'item' => 'basic',
        'expected' => 'Basic Plan - $ 10.00',
      ],
      'item with amount but no label' => [
        'element' => [
          '#items' => [
            'no_label' => [
              'amount' => 25.50,
            ],
          ],
        ],
        'item' => 'no_label',
        'expected' => '$ 25.50',
      ],
      'item without amount returns key' => [
        'element' => [
          '#items' => [
            'no_amount' => [
              'label' => 'Free Plan',
            ],
          ],
        ],
        'item' => 'no_amount',
        'expected' => 'no_amount',
      ],
      'item with custom formatting' => [
        'element' => [
          '#amount_prefix' => '€ ',
          '#amount_suffix' => ' EUR',
          '#amount_precision' => 0,
          '#items' => [
            'premium' => [
              'label' => 'Premium Plan',
              'amount' => 99.99,
            ],
          ],
        ],
        'item' => 'premium',
        'expected' => 'Premium Plan - € 100 EUR',
      ],
    ];
  }

  /**
   * Tests getElementTotal method.
   *
   * @covers ::getElementTotal
   * @dataProvider providerGetElementTotal
   */
  public function testGetElementTotal(array $element, array|string|null $values, float $expected): void {
    $result = WebformPaymentElement::getElementTotal($element, $values);
    $this->assertEquals($expected, $result);
  }

  /**
   * Data provider for testGetElementTotal.
   *
   * @return array
   *   Test cases with element config, values, and expected total.
   */
  public static function providerGetElementTotal(): array {
    return [
      'null values returns zero' => [
        'element' => [
          '#items' => [],
        ],
        'values' => NULL,
        'expected' => 0.0,
      ],
      'single value as string' => [
        'element' => [
          '#items' => [
            'item1' => ['amount' => 10.00],
          ],
        ],
        'values' => 'item1',
        'expected' => 10.0,
      ],
      'single value as array' => [
        'element' => [
          '#items' => [
            'item1' => ['amount' => 15.50],
          ],
        ],
        'values' => ['item1'],
        'expected' => 15.5,
      ],
      'multiple values' => [
        'element' => [
          '#items' => [
            'item1' => ['amount' => 10.00],
            'item2' => ['amount' => 20.00],
            'item3' => ['amount' => 5.50],
          ],
        ],
        'values' => ['item1', 'item2', 'item3'],
        'expected' => 35.5,
      ],
      'empty array returns zero' => [
        'element' => [
          '#items' => [
            'item1' => ['amount' => 10.00],
          ],
        ],
        'values' => [],
        'expected' => 0.0,
      ],
      'non-existent item key returns zero' => [
        'element' => [
          '#items' => [
            'item1' => ['amount' => 10.00],
          ],
        ],
        'values' => ['missing_item'],
        'expected' => 0.0,
      ],
      'mixed valid and invalid items' => [
        'element' => [
          '#items' => [
            'item1' => ['amount' => 10.00],
            'item2' => ['amount' => 20.00],
          ],
        ],
        'values' => ['item1', 'missing', 'item2'],
        'expected' => 30.0,
      ],
      'item without amount treated as zero' => [
        'element' => [
          '#items' => [
            'item1' => ['label' => 'No Amount'],
            'item2' => ['amount' => 15.00],
          ],
        ],
        'values' => ['item1', 'item2'],
        'expected' => 15.0,
      ],
      'decimal amounts' => [
        'element' => [
          '#items' => [
            'item1' => ['amount' => 10.99],
            'item2' => ['amount' => 20.01],
            'item3' => ['amount' => 5.50],
          ],
        ],
        'values' => ['item1', 'item2', 'item3'],
        'expected' => 36.5,
      ],
      'large amounts' => [
        'element' => [
          '#items' => [
            'item1' => ['amount' => 1000.00],
            'item2' => ['amount' => 2500.50],
          ],
        ],
        'values' => ['item1', 'item2'],
        'expected' => 3500.5,
      ],
    ];
  }

}
