<?php

namespace Drupal\webform_qr_code_element\Element;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\webform\Element\WebformComputedInterface;
use Drupal\webform\Element\WebformComputedToken;
use Drupal\webform\Plugin\WebformElementDisplayOnInterface;
use Drupal\webform\WebformSubmissionInterface;
use Endroid\QrCode\Builder\Builder;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\ErrorCorrectionLevel;
use Endroid\QrCode\RoundBlockSizeMode;
use Symfony\Component\Routing\Exception\InvalidParameterException;

/**
 * @FormElement("qr_code")
 */
class QrCodeElement extends WebformComputedToken {

  /**
   * Add HTML markup to show image of QR code.
   */
  public static function processWebformComputed(&$element, FormStateInterface $form_state, &$complete_form) {
    parent::processWebformComputed($element, $form_state, $complete_form);
    $text = $element['value']['#markup'];
    $element['value'] = self::renderImage($text, $element);

    return $element;
  }

  /**
   * Build the HTML image tag with the QR code as base64 data URI.
   */
  public static function renderImage(string $text, array $element): array {
    $builder = self::getBuilder($element, $text, FALSE);

    $alt = htmlentities($element['#title'] ?? '');
    $dataUri = $builder->build()->getDataUri();

    return ['#markup' => Markup::create(sprintf('<img alt="%s" src="data:%s">', $alt, $dataUri))];
  }

  /**
   * @param array $element
   * @param string $text
   * @param bool $is_pdf
   *
   * @return \Endroid\QrCode\Builder\Builder
   */
  public static function getBuilder(array $element, string $text, bool $is_pdf): Builder {
    if (!$text) {
      throw new \InvalidArgumentException(t("This QR code has no text. Check the :setting setting.", [':setting' => t('Text to store in the QR code')]));
    }
    $show_label_below_code = self::getSetting($element, 'show_label_below_code');

    return new Builder(
      writerOptions: [],
      validateResult: FALSE,
      data: $text,
      encoding: new Encoding('UTF-8'),
      errorCorrectionLevel: ErrorCorrectionLevel::High,
      size: $is_pdf ? self::getSetting($element, 'pdf_qr_size') : 280,
      margin: 1,
      roundBlockSizeMode: RoundBlockSizeMode::Margin,
      labelText: $show_label_below_code ? $text : ''
    );
  }

  /**
   * Missing values are replaced by their default.
   */
  public static function getSetting(array $element, string $key) {
    return $element["#$key"] ?? self::getDefaults()[$key];
  }

  /**
   * Each setting needs to have a default in order to show up in the form.
   * In the webform config entity, only keys that differ from the default are present.
   */
  public static function getDefaults(): array {
    return [
      'attach_pdf_to_emails' => FALSE,
      'attach_png_to_emails' => FALSE,
      'display_on' => WebformElementDisplayOnInterface::DISPLAY_ON_VIEW,
      'mode' => WebformComputedInterface::MODE_TEXT,
      'pdf_attachment_filename' => "Ticket.pdf",
      'pdf_qr_size' => 160,
      'pdf_qr_x' => 20,
      'pdf_qr_y' => 40,
      'pdf_template' => NULL,
      'show_label_below_code' => FALSE,
      'store' => TRUE,
      'template_format' => '',
      'textfields' => [],
      'title' => t('QR code for admission'),
    ];
  }

  /**
   * Computes the 'data' that will get encoded into the QR code.
   */
  public static function computeValue(array $element, WebformSubmissionInterface $webform_submission) {
    if (strpos($element['#template'], 'token-view-url') > 0) {
      if (!$webform_submission->id()) {
        return "not enough context";
      }
    }
    try {
      return parent::computeValue($element, $webform_submission);
    }
    catch (InvalidParameterException $e) {
      return "Missing context: " . $e->getMessage();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    $class = get_class($this);
    $process = [
      '#process' => [
        [$class, 'processWebformComputed'],
      ],
    ];

    return $process + parent::getInfo();
  }

}
