<?php

declare(strict_types=1);

namespace Drupal\webform_qr_code_element;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\Serialization\Yaml;
use Drupal\file\Entity\File;
use Drupal\webform\Entity\WebformSubmission;
use Drupal\webform\Twig\WebformTwigExtension;
use Drupal\webform\WebformInterface;
use Drupal\webform\WebformSubmissionInterface;
use Drupal\webform\WebformTokenManagerInterface;
use Drupal\webform_qr_code_element\Element\QrCodeElement;
use Drupal\webform_qr_code_element\Element\TextPosition;
use Endroid\QrCode\Exception\BlockSizeTooSmallException;
use Endroid\QrCode\Exception\ValidationException;
use Endroid\QrCode\Writer\PdfWriter;
use Endroid\QrCode\Writer\Result\ResultInterface;
use setasign\Fpdi\Fpdi;
use setasign\Fpdi\FpdiException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * This service checks for QR code elements in webform submissions and conditionally
 * generates email attachments based on the element's configuration settings.
 */
class AttachmentService extends ControllerBase {

  private WebformSubmissionInterface $submission;

  private ModuleHandlerInterface $module_handler;

  private string $value = '';

  /**
   * @var \Drupal\webform\WebformTokenManagerInterface
   */
  private WebformTokenManagerInterface $token_manager;

  public function __construct(ModuleHandlerInterface $module_handler, WebformTokenManagerInterface $token_manager) {
    $this->module_handler = $module_handler;
    $this->token_manager = $token_manager;
  }

  /**
   * Checks if the submission’s webform has a QR code element.
   */
  public function getQrCodeElement(WebformSubmissionInterface $submission): array {
    $webform = $submission->getWebform();
    foreach ($webform->getElementsInitializedAndFlattened() as $key => $element) {
      if ($element['#type'] === 'qr_code') {
        $this->submission = $submission;
        $this->value = $submission->getElementData($key);

        return $element;
      }
    }

    return [];
  }

  /**
   * Perform the token calculation for the 'Test' button on the config form.
   */
  public function computeValueForTestButton($element, WebformInterface $webform): void {
    $submission = $this->getLatestWebformSubmission($webform);
    if (!$submission) {
      $submission = $this->createRandomSubmission($webform);
    }
    $this->submission = $submission;
    $this->value = QrCodeElement::computeValue($element, $submission);
  }

  /**
   * Find the webform submission with the largest sid for a given webform.
   *
   * @return \Drupal\webform\WebformSubmissionInterface|null
   *   The webform submission with the largest sid, or NULL if none found.
   */
  private function getLatestWebformSubmission(WebformInterface $webform): ?WebformSubmissionInterface {
    $query = \Drupal::entityQuery('webform_submission')
      ->condition('webform_id', $webform->id())
      ->sort('sid', 'DESC')
      ->range(0, 1)
      ->accessCheck(FALSE);
    $sids = $query->execute();

    if (empty($sids)) {
      return NULL;
    }

    return WebformSubmission::load(reset($sids));
  }

  /**
   * Attaches files to emails if configured in the form element.
   */
  public function attachQrCode(array $element, array &$message): void {
    if (!empty(QrCodeElement::getSetting($element, 'attach_png_to_emails'))) {
      $this->ensureEmailAttachmentHelperIsEnabled();
      $message['params']['attachments'][] = [
        'filecontent' => QrCodeElement::getBuilder($element, $this->value, FALSE)->build()->getString(),
        'filename' => t('QR code') . '.png',
        'filemime' => 'image/png',
      ];
    }

    if (!empty(QrCodeElement::getSetting($element, 'attach_pdf_to_emails'))) {
      $filename = QrCodeElement::getSetting($element, 'pdf_attachment_filename');
      try {
        $this->ensureEmailAttachmentHelperIsEnabled();
        $pdf = $this->getPdf($element);
        $message['params']['attachments'][] = [
          'filecontent' => $pdf->getString(),
          'filename' => $filename,
          'filemime' => 'application/pdf',
        ];
      }
      catch (ValidationException | FpdiException | BlockSizeTooSmallException $e) {
        $msg = t('Could not attach the PDF') . ': ' . $e->getMessage();
        \Drupal::logger('webform_qr_code_element')->error($msg);
        \Drupal::messenger()->addError($msg);
      }
    }
  }

  /**
   * Checks if the needed Drupal module is usable.
   * 'email_attachment' implements a mail_hook to convert the array
   * to MIME-separated multipart email.
   */
  public function ensureEmailAttachmentHelperIsEnabled(): void {
    if (!$this->module_handler->moduleExists('email_attachment')) {
      $msg = t(
        'The module @url is required for attaching QR codes.', [
          '@url' => Markup::create(
            '<a href="https://www.drupal.org/project/email_attachment">Email Attachment Helper</a>',
          ),
        ]
      );
      \Drupal::logger('webform_qr_code_element')->error($msg);
      \Drupal::messenger()->addError($msg);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(?ContainerInterface $container = NULL): self {
    if (!$container) {
      $container = \Drupal::getContainer();
    }

    return new self(
      $container->get('module_handler'),
      $container->get('webform.token_manager')
    );
  }

  /**
   * Builds a PDF in memory.
   *
   * @param array $element
   *
   * @return \Endroid\QrCode\Writer\Result\ResultInterface
   *
   * @throws \Endroid\QrCode\Exception\ValidationException
   *
   * @throws \setasign\Fpdi\PdfParser\CrossReference\CrossReferenceException
   * @throws \setasign\Fpdi\PdfParser\Filter\FilterException
   * @throws \setasign\Fpdi\PdfParser\PdfParserException
   * @throws \setasign\Fpdi\PdfParser\Type\PdfTypeException
   * @throws \setasign\Fpdi\PdfReader\PdfReaderException
   * @throws \Endroid\QrCode\Exception\BlockSizeTooSmallException
   */
  public function getPdf(array $element): ResultInterface {
    $builder = QrCodeElement::getBuilder($element, $this->value, TRUE);

    if (($fids = QrCodeElement::getSetting($element, 'pdf_template')) && !empty($fids[0])) {
      $pdf = new Fpdi();
      $file = File::load($fids[0]);
      $page_count = $pdf->setSourceFile($file->getFileUri());
      $pdf->AddPage();
      $pdf->useTemplate($pdf->importPage(1));
    }
    else {
      $pdf = new \FPDF();
      $page_count = 1;
      $pdf->AddPage();
    }
    $pdf->SetAutoPageBreak(FALSE);
    $textfields = QrCodeElement::getSetting($element, 'textfields');
    for ($page = 1; $page <= $page_count; $page++) {
      $this->addTextFieldsToPage($textfields, $pdf, $page);
      if ($page < $page_count) {
        $pdf->AddPage();
        $pdf->useTemplate($pdf->importPage($page + 1));
      }
    }

    return $builder->build(
      new PdfWriter(), [
        PdfWriter::WRITER_OPTION_PDF => $pdf,
        PdfWriter::WRITER_OPTION_X => QrCodeElement::getSetting($element, 'pdf_qr_x'),
        PdfWriter::WRITER_OPTION_Y => QrCodeElement::getSetting($element, 'pdf_qr_y'),
      ]
    );
  }

  /**
   * Add customized, user-defined text to the PDF.
   *
   * @param array<int, array<string, bool|int|string>> $items
   *   Information on the textfields, as defined by the "text_position" form element.
   */
  protected function addTextFieldsToPage(array $items, Fpdi|\FPDF $pdf, int $current_page): void {
    foreach ($items as $item) {
      $target_page = TextPosition::getSetting($item, 'page');
      if ($target_page != $current_page) {
        continue;
      }
      $font_size = TextPosition::getSetting($item, 'font_size');
      $pdf->SetFont(
        TextPosition::getSetting($item, 'font_name'),
        $this->calculateStyle($item),
        $font_size
      );
      $x = TextPosition::getSetting($item, 'x');
      $pdf->SetXY(
        $x,
        TextPosition::getSetting($item, 'y')
      );
      $pdf->SetLeftMargin($x);
      $twig_text = TextPosition::getSetting($item, 'twig_text');
      if (!$twig_text) {
        $text = TextPosition::getSetting($item, 'text');
      }
      else {
        $text = WebformTwigExtension::renderTwigTemplate(
          $this->submission,
          $twig_text
        );
      }
      $text = $this->token_manager->replace($text, $this->submission);
      $pdf->Write($font_size / 3, iconv('UTF-8', 'windows-1252', (string) $text));
    }
  }

  /**
   * Produces the format that FPDF needs for the SetFont call.
   */
  private function calculateStyle(array $item): string {
    $style = '';
    if (TextPosition::getSetting($item, 'bold')) {
      $style .= 'B';
    }
    if (TextPosition::getSetting($item, 'underlined')) {
      $style .= 'U';
    }
    if (TextPosition::getSetting($item, 'italic')) {
      $style .= 'I';
    }

    return $style;
  }

  /**
   * Builds the response.
   */
  public function downloadPdf(WebformInterface $webform, string $filename): Response {
    $pdfContent = file_get_contents($this->getPdfFilenameForTestButton());
    if (!$pdfContent) {
      return new Response("File not found. Run `QrCode::buildPdfTemplate` first.", 400);
    }
    $response = new Response($pdfContent);
    $response->headers->set('Content-Type', 'application/pdf');
    $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '"');
    $response->headers->set('Content-Length', (string) strlen($pdfContent));

    return $response;
  }

  /**
   *
   */
  public function getPdfFilenameForTestButton(): string {
    $public_files_uri = \Drupal::service('file_system')->realpath('public://');

    return "$public_files_uri/webform_qr_code_element.pdf";
  }

  /**
   * Create, but don’t save a submission for the PDF test button.
   */
  private function createRandomSubmission(WebformInterface $webform): WebformSubmission {
    /** @var \Drupal\webform\WebformSubmissionGenerate $webformSubmissionGenerate */
    $webformSubmissionGenerate = \Drupal::service('webform_submission.generate');

    return \Drupal::entityTypeManager()->getStorage('webform_submission')->create([
      'webform_id' => $webform->id(),
      'data' => Yaml::encode($webformSubmissionGenerate->getData($webform)),
    ]);
  }

}
