<?php

namespace Drupal\webform_qr_code_element\Element;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element\FormElementBase;
use Drupal\webform\Twig\WebformTwigExtension;

/**
 * Provides a compound form element for positioned text with formatting.
 *
 * @FormElement("text_position")
 */
class TextPosition extends FormElementBase {

  /**
   * Processes the text position element.
   */
  public static function processTextPosition(&$element, FormStateInterface $form_state, &$complete_form) {
    $value = $element['#default_value'] ?? [];

    $element['#tree'] = TRUE;
    $text = t('Text');
    $element['text'] = [
      '#type' => 'textarea',
      '#title' => $text,
      '#required' => TRUE,
      '#default_value' => $value['text'] ?? '',
      '#token_types' => ['webform', 'webform_submission', 'webform_handler', 'site', 'date'],
      '#token_help' => TRUE,
      '#rows' => 2,
    ];
    $twig_text = $value['twig_text'] ?? '';
    $element['twig_text'] = [
      '#type' => 'details',
      '#open' => $twig_text ? TRUE : NULL,
      '#title' => t('Override :text with Twig', [':text' => $text]),
      'template' => [
        '#title' => t('Allows more complex logic such as loops and conditions'),
        '#description' => t('If set, the “:text” above will be ignored and this is used instead.', [':text' => $text]),
        '#type' => 'webform_codemirror',
        '#default_value' => $twig_text,
        '#mode' => 'twig',
      ],
      'help' => WebformTwigExtension::buildTwigHelp(),
    ];

    $element['position'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['text-position-coordinates']],
    ];
    $element['position']['x'] = [
      '#type' => 'number',
      '#title' => t('X position'),
      '#required' => TRUE,
      '#default_value' => $value['x'] ?? 0,
      '#field_suffix' => 'mm',
      '#size' => 8,
      '#step' => 0.1,
      '#min' => 0,
      '#max' => 210,
    ];
    $element['position']['y'] = [
      '#type' => 'number',
      '#title' => t('Y position'),
      '#required' => TRUE,
      '#default_value' => $value['y'] ?? 0,
      '#field_suffix' => 'mm',
      '#size' => 8,
      '#step' => 0.1,
      '#min' => 0,
      '#max' => 297,
    ];
    $element['position']['page'] = [
      '#type' => 'number',
      '#title' => t('Page'),
      '#required' => TRUE,
      '#default_value' => $value['page'] ?? 1,
      '#size' => 7,
      '#step' => 1,
      '#min' => 0,
    ];

    // Font settings container.
    $element['font'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['text-position-font']],
    ];

    $element['font']['font_name'] = [
      '#type' => 'select',
      '#title' => t('Font'),
      '#options' => [
        'Arial' => 'Arial',
        'Courier' => 'Courier',
        'Times' => 'Times',
      ],
      '#default_value' => $value['font_name'] ?? 'Arial',
    ];

    $element['font']['font_size'] = [
      '#type' => 'number',
      '#title' => t('Size'),
      '#required' => TRUE,
      '#default_value' => $value['font_size'] ?? 16,
      '#size' => 6,
      '#step' => 1,
      '#min' => 1,
      '#max' => 72,
    ];

    // Text formatting options.
    /** @var \Drupal\Core\Theme\ThemeManagerInterface $theme_manager */
    $theme_manager = \Drupal::service('theme.manager');
    $theme = $theme_manager->getActiveTheme()->getName();
    $element['formatting'] = [
      '#type' => str_contains($theme, 'claro') ? 'details' : 'container',
      '#attributes' => ['class' => ['text-position-formatting']],
      '#title' => t('Text formatting'),
    ];

    $element['formatting']['bold'] = [
      '#type' => 'checkbox',
      '#title' => t('Bold'),
      '#default_value' => $value['bold'] ?? FALSE,
    ];

    $element['formatting']['italic'] = [
      '#type' => 'checkbox',
      '#title' => t('Italic'),
      '#default_value' => $value['italic'] ?? FALSE,
    ];

    $element['formatting']['underlined'] = [
      '#type' => 'checkbox',
      '#title' => t('Underlined'),
      '#default_value' => $value['underlined'] ?? FALSE,
    ];

    $element['#attached']['library'][] = 'webform_qr_code_element/text_position_element';

    return $element;
  }

  /**
   * Reshuffle the submitted data to a flat array for easier upstream use and encapsulation.
   */
  public static function validateTextPosition(&$element, FormStateInterface $form_state) {
    $values = $form_state->getValue($element['#parents']);

    $processed_value = [
      'text' => $values['text'],
      'twig_text' => $values['twig_text']['template'],
      'x' => (float) str_replace(",", ".", $values['position']['x']),
      'y' => (float) str_replace(",", ".", $values['position']['y']),
      'page' => (int) $values['position']['page'],
      'font_name' => $values['font']['font_name'],
      'font_size' => (int) $values['font']['font_size'],
      'bold' => (bool) $values['formatting']['bold'],
      'italic' => (bool) $values['formatting']['italic'],
      'underlined' => (bool) $values['formatting']['underlined'],
    ];
    foreach (self::getDefaults() as $key => $default) {
      if ($processed_value[$key] === $default) {
        unset($processed_value[$key]);
      }
    }

    $form_state->setValueForElement($element, $processed_value);
  }

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    $class = get_class($this);

    return [
      '#input' => TRUE,
      '#process' => [
        [$class, 'processTextPosition'],
      ],
      '#element_validate' => [
        [$class, 'validateTextPosition'],
      ],
      '#theme_wrappers' => ['form_element'],
      '#default_value' => $this->getDefaults(),
    ];
  }

  /**
   * Retrieves a setting value from an element configuration array.
   *
   * If the requested key is not present in the element array, returns the
   * corresponding default value instead.
   *
   * @param array<string, bool|float|int|string> $element
   *   The element configuration array.
   * @param string $key
   *   The setting key to retrieve.
   *
   * @return bool|float|int|string The setting value or its default if not found
   */
  public static function getSetting(array $element, string $key) {
    return $element[$key] ?? self::getDefaults()[$key];
  }

  /**
   * @return array
   */
  protected static function getDefaults(): array {
    return [
      'text' => '',
      'twig_text' => '',
      'x' => 0.0,
      'y' => 0.0,
      'page' => 1,
      'font_name' => 'Arial',
      'font_size' => 16,
      'bold' => FALSE,
      'italic' => FALSE,
      'underlined' => FALSE,
    ];
  }

}
