<?php

namespace Drupal\webform_qr_code_element\Plugin\WebformElement;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\file\Entity\File;
use Drupal\webform\Plugin\WebformElement\WebformComputedToken;
use Drupal\webform\WebformInterface;
use Drupal\webform\WebformSubmissionInterface;
use Drupal\webform_qr_code_element\AttachmentService;
use Drupal\webform_qr_code_element\Element\QrCodeElement;
use Endroid\QrCode\Exception\BlockSizeTooSmallException;
use setasign\Fpdi\Fpdi;
use setasign\Fpdi\PdfParser\PdfParserException;
use Symfony\Component\HttpFoundation\Response;

/**
 * @WebformElement(
 *   id = "qr_code",
 *   label = @Translation("QR Code"),
 *   description = @Translation("A computed webform element that generates a QR code containing customizable token-based content. The QR code is only visible after form submission (not during form completion). The element allows form authors to embed submission IDs, secure tokens, or other dynamic values that can be scanned by external applications for participant lookup and verification. Allows to attach customizable PDFs to emails, for example 'Ticket.pdf'."),
 *   category = @Translation("Computed Elements"),
 * )
 */
class QrCode extends WebformComputedToken {

  /**
   * Ajax callback to download a preview PDF.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   An HTTP response with a download.
   */
  public static function ajaxPdfPreview(array &$form, FormStateInterface $form_state): Response {
    $response = new AjaxResponse();

    try {
      $parameterBag = \Drupal::routeMatch()->getParameters();
      $webform = $parameterBag->get('webform');
      self::savePdfForTestButton($form_state, $webform);
      $key = 'pdf_attachment_filename';
      $new_url = \Drupal::service('url_generator')->generateFromRoute(
        'webform_qr_code_element.download_test_pdf',
        [
          'filename' => QrCodeElement::getSetting(["#$key" => $form_state->getValues()['properties'][$key]], $key),
          'webform' => $webform->id(),
          'key' => $parameterBag->get('key'),
        ]
      );
      $response->addCommand(new RedirectCommand($new_url));
    }
    catch (BlockSizeTooSmallException $e) {
      $msg = t('Please reduce the length of the text to store in the QR code or increase its size.');
      foreach (['test-button', 'form-item--properties-pdf-qr-size'] as $class) {
        $response->addCommand(new MessageCommand($msg, ".$class", ['type' => 'error']));
      }
    }
    catch (\Throwable $e) {
      $response->addCommand(new MessageCommand($e->getMessage(), '.test-button', ['type' => 'error']));
    }

    return $response;
  }

  /**
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   * @param \Drupal\webform\WebformInterface $webform
   *
   * @throws \Endroid\QrCode\Exception\ValidationException
   * @throws \setasign\Fpdi\PdfParser\CrossReference\CrossReferenceException
   * @throws \setasign\Fpdi\PdfParser\Filter\FilterException
   * @throws \setasign\Fpdi\PdfParser\PdfParserException
   * @throws \setasign\Fpdi\PdfParser\Type\PdfTypeException
   * @throws \setasign\Fpdi\PdfReader\PdfReaderException
   */
  private static function savePdfForTestButton(FormStateInterface $form_state, WebformInterface $webform): void {
    $service = AttachmentService::create();
    $element = [];
    foreach ($form_state->getValues()['properties'] as $key => $value) {
      $element["#$key"] = $value;
    }
    $service->computeValueForTestButton($element, $webform);
    $service->getPdf($element)->saveToFile($service->getPdfFilenameForTestButton());
  }

  /**
   * Ajax callback for textfields operations.
   */
  public static function textFieldsAjaxCallback(array &$form) {
    return $form['properties']['attachments']['pdf_container']['textfields'];
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);
    $desc = t('For entry tickets, this needs to be unique and not guessable.') . '<br>' .
      t('Examples') . ':<br>' .
      '• [webform_submission:sid]-[webform_submission:created]<br>' .
      '• [webform_submission:token]<br><br>' .
      t('For SEPA credit transfer QR codes, see @ECP.', ['@ECP' => Markup::create('<a href="https://en.wikipedia.org/wiki/EPC_QR_code">ECP</a>')]);
    $form['computed']['template'] = [
      '#type' => 'textarea',
      '#required' => TRUE,
      '#help' => '',
      '#title' => t('Text to store in the QR code'),
      '#rows' => 2,
      '#description_display' => 'after',
      '#description' => $desc,
      '#element_validate' => [
        [
          '\Drupal\webform\WebformTokenManager',
          'validateElement',
        ],
      ],
      '#token_types' => [
        'webform',
        'webform_submission',
        'webform_handler',
        'site',
        'date',
      ],
    ];

    $form['image_settings'] = [
      '#type' => 'fieldset',
      '#title' => t('Image settings'),
      'show_label_below_code' => [
        '#type' => 'checkbox',
        '#return_value' => TRUE,
        '#title' => t('Show label below QR code'),
        '#description' => t('This will show the text that is encoded in the QR code.') . ' ' .
        t('Helpful when visitors need to see or manually enter the URL.'),
      ],
    ];
    $form['attachments'] = [
      '#type' => 'fieldset',
      '#title' => t('Email attachments'),
      '#description' => 'If you have configured an email to be sent when the form is submitted, you can attach the QR code by checking a box below.',
      'attach_png_to_emails' => [
        '#type' => 'checkbox',
        '#return_value' => TRUE,
        '#title' => t('PNG image'),
      ],
      'attach_pdf_to_emails' => [
        '#type' => 'checkbox',
        '#return_value' => TRUE,
        '#title' => t('PDF document'),
      ],
      'pdf_container' => [
        '#type' => 'container',
        '#states' => [
          'visible' => [
            '[name="properties[attach_pdf_to_emails]"]' => ['checked' => TRUE],
          ],
        ],
        'pdf_template' => [
          '#type' => 'managed_file',
          '#upload_location' => 'public://',
          '#upload_validators' => [
            'file_validate_extensions' => ['pdf'],
          ],
          '#title' => t('PDF file template') . ' (' . t('optional') . ')',
        ],
        'pdf_attachment_filename' => [
          '#type' => 'textfield',
          '#title' => t('PDF attachment file name'),
        ],
        'pdf_qr_size' => [
          '#type' => 'number',
          '#title' => t('Size of the QR code in mm'),
          '#min' => 21,
          '#max' => 210,
          '#step' => 1,
          '#field_suffix' => 'mm',
          '#wrapper_attributes' => ['class' => ['form-item--properties-pdf-qr-size']],
        ],
        'pdf_qr_x' => [
          '#type' => 'number',
          '#title' => t('X position in mm from the left page margin'),
          '#min' => 0,
          '#max' => 210,
          '#step' => 1,
          '#field_suffix' => 'mm',
        ],
        'pdf_qr_y' => [
          '#type' => 'number',
          '#title' => t('Y position in mm from the top page margin'),
          '#min' => 0,
          '#max' => 297,
          '#step' => 1,
          '#field_suffix' => 'mm',
        ],
        'textfields' => [
          '#type' => 'webform_multiple',
          '#title' => t('Additional texts'),
          '#description' => t('This field supports tokens.'),
          '#empty_items' => 0,
          '#element' => [
            '#type' => 'text_position',
          ],
        ],
        'actions' => [
          '#type' => 'actions',
          '#access' => TRUE,
          '#attributes' => ['class' => ['test-button']],
          'test' => [
            '#type' => 'button',
            '#value' => t('Show test PDF'),
            '#button_type' => 'secondary',
            '#ajax' => [
              'callback' => [static::class, 'ajaxPdfPreview'],
              'event' => 'click',
            ],
            '#access' => TRUE,
          ],
        ],
      ],
    ];

    // Inherited from base class, but not needed:
    unset(
      $form['computed']['ajax'],
      $form['computed']['hide_empty'],
      $form['computed']['mode'],
      $form['computed']['warning'],
      $form['computed']['whitespace'],
      $form['form']['prepopulate']
    );

    return $form;
  }

  /**
   * Validate user input.
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    $attach_pdf = $form_state->getValue('attach_pdf_to_emails');
    if ($attach_pdf || $form_state->getValue('attach_png_to_emails')) {
      AttachmentService::create()->ensureEmailAttachmentHelperIsEnabled();
    }

    $pdf_templates = $form_state->getValue('pdf_template');
    if ($attach_pdf && $pdf_templates) {
      $this->validatePdfTemplate($pdf_templates, $form['properties']['attachments']['pdf_container']['pdf_template'], $form_state);
    }

    $filename = $form_state->getValue('pdf_attachment_filename');
    if ($filename) {
      $routing = Yaml::decode(file_get_contents(__DIR__ . '/../../../webform_qr_code_element.routing.yml'));
      $regex = $routing['webform_qr_code_element.download_test_pdf']['requirements']['filename'];
      if (!preg_match("/^$regex\$/", $filename)) {
        $form_state->setErrorByName('pdf_attachment_filename', t('The filename can only contain these characters:') . ' ' . $regex);
      }
    }
  }

  /**
   * @param int[] $pdf_template_fids
   * @param array $element
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  protected function validatePdfTemplate(array $pdf_template_fids, array $element, FormStateInterface $form_state): void {
    $file = File::load(reset($pdf_template_fids));
    $file_name = $file->getFileUri();
    $pdf = new Fpdi();
    try {
      $pdf->setSourceFile($file_name);

      // We could parse it, so keep it.
      $file->setPermanent();
      $file->save();
    }
    catch (PdfParserException | \InvalidArgumentException $e) {
      $form_state->setError($element, $e->getMessage());

      return;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function formatHtml(array $element, WebformSubmissionInterface $webform_submission, array $options = []) {
    $value = $this->getValue($element, $webform_submission, $options);

    return QrCodeElement::renderImage($value, $element)['#markup'];
  }

  /**
   * {@inheritdoc}
   */
  protected function defineDefaultProperties() {
    return QrCodeElement::getDefaults() + parent::defineDefaultProperties();
  }

}
