<?php

namespace Drupal\webform_registration_handler\Plugin\WebformHandler;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\user\RoleInterface;
use Drupal\user\UserInterface;
use Drupal\webform\Plugin\WebformHandlerBase;
use Drupal\webform\WebformSubmissionInterface;
use Drupal\webform_registration_handler\CivicrmManagementInterface;
use Drupal\webform_registration_handler\Exception\InvalidUserException;
use Drupal\webform_registration_handler\Exception\UserExistsException;
use Drupal\webform_registration_handler\UserManagementInterface;
use Drupal\webform_registration_handler\WebformHelperInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * User Registration Webform Handler plugin.
 *
 * @WebformHandler(
 *   id = "user_registration_webform_handler",
 *   label = @Translation("User Registration"),
 *   category = @Translation("User"),
 *   description = @Translation("Handles webform submissions to create user accounts upon submission."),
 *   cardinality = \Drupal\webform\Plugin\WebformHandlerInterface::CARDINALITY_SINGLE,
 *   results = \Drupal\webform\Plugin\WebformHandlerInterface::RESULTS_PROCESSED,
 *   submission = \Drupal\webform\Plugin\WebformHandlerInterface::SUBMISSION_REQUIRED,
 * )
 */
class UserRegistrationWebformHandler extends WebformHandlerBase {

  /**
   * Webform helper.
   */
  private \Closure $webformHelper;

  /**
   * Exception logger.
   */
  private LoggerChannelInterface $exceptionLogger;

  /**
   * User management.
   */
  private UserManagementInterface $userManagement;

  /**
   * Current user.
   */
  protected AccountInterface $currentUser;

  /**
   * Civicrm management.
   */
  private \Closure $civicrmManagement;

  /**
   * Account.
   */
  protected ?UserInterface $account = NULL;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $plugin = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $plugin->exceptionLogger = $container->get('webform_registration_handler.exception_logger');
    $plugin->userManagement = $container->get(UserManagementInterface::class);
    $plugin->currentUser = $container->get(AccountInterface::class);
    $plugin->webformHelper = fn(): WebformHelperInterface => $container->get(WebformHelperInterface::class);
    $plugin->civicrmManagement = fn(): CivicrmManagementInterface => $container->get(CivicrmManagementInterface::class);

    return $plugin;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'create_user' => [
        'selected_roles' => [],
        'sync_with_civicrm' => FALSE,
        'email_verification_message' => $this->t('Your user has been created. You have been sent an email to verify your address, click the link in the email to log in and set your password.'),
        'mapping' => [
          'user_field_mapping' => [],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {

    $roles = $this->userManagement->getListOfRoleNames();

    $form['create_user'] = [
      '#type' => 'details',
      '#title' => $this->t('User creation'),
    ];

    $form['create_user']['email_verification'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Require email verification when a visitor creates an account.'),
      '#description' => $this->t("New users will be required to validate their email address and set the password prior to logging into the site. If unchecked, users can set their password in the webform during account creation."),
      '#default_value' => $this->configuration['create_user']['email_verification'] ?? TRUE,
      '#disabled' => TRUE,
    ];

    $form['create_user']['email_verification_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Email Verification Message'),
      '#description' => $this->t('This message will be displayed to users upon successful registration. There is a default message below that you can customize.'),
      '#rows' => 4,
      '#maxlength' => 255,
      '#required' => TRUE,
      '#placeholder' => $this->configuration['create_user']['email_verification_message'],
      '#default_value' => $this->configuration['create_user']['email_verification_message'],
    ];

    $roleOptions = [];
    foreach ($roles as $role) {
      $roleOptions[$role->id()] = $role->label();
    }

    $form['create_user']['selected_roles'] = [
      '#type' => 'checkboxes',
      '#options' => $roleOptions,
      '#required' => FALSE,
      '#default_value' => $this->configuration['create_user']['selected_roles'],
      '#description' => $this->t('Select the roles that should be added to the newly created user.'),
      '#access' => $this->currentUser->hasPermission('administer permissions'),
    ];

    $form['create_user']['selected_roles'][RoleInterface::AUTHENTICATED_ID] = [
      '#disabled' => TRUE,
      '#checked' => TRUE,
      '#default_value' => TRUE,
    ];

    $form['create_user']['sync_with_civicrm'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Synchronize with CiviCRM'),
      '#default_value' => $this->configuration['create_user']['sync_with_civicrm'],
      '#description' => $this->t('Check this box to synchronize the newly created Drupal user with a CiviCRM contact having the same email address, if CiviCRM is installed.'),
    ];

    $form['create_user']['mapping'] = [
      '#type' => 'details',
      '#title' => $this->t('User field mapping'),
    ];

    $form['create_user']['mapping']['user_field_mapping'] = [
      '#type' => 'webform_mapping',
      '#description' => $this->t('Map webform element values to user fields.'),
      '#required' => FALSE,
      '#source' => ($this->webformHelper)()->getMappingSourceFields(),
      '#destination' => ($this->webformHelper)()->getMappingDestinationFields($this->webform),
      '#default_value' => $this->configuration['create_user']['mapping']['user_field_mapping'],
      '#source__title' => $this->t('User field destination'),
      '#destination__type' => 'select',
      '#destination__title' => $this->t('Webform element'),
      '#destination__description' => NULL,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    $this->applyFormStateToConfiguration($form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);

    $userFieldMapping = $form_state->getValue([
      'create_user',
      'mapping',
      'user_field_mapping',
    ]);

    $requiredMappingKeys = ($this->webformHelper)()->getRequiredMappingFields();
    foreach ($requiredMappingKeys as $key) {
      if (empty($userFieldMapping[$key])) {
        $form_state->setErrorByName('[create_user][mapping][user_field_mapping]', $this->t('User field mapping is incomplete.'));
        break;
      }
    }

    $selectedRoles = $form_state->getValue(['create_user', 'selected_roles']);
    if (empty(array_filter($selectedRoles))) {
      $form_state->setErrorByName('[create_user][roles][selected_roles]', $this->t('Please select at least one role.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state, WebformSubmissionInterface $webform_submission) {
    parent::validateForm($form, $form_state, $webform_submission);

    $webformSubmissionData = $this->getWebformUserData($webform_submission);
    if (empty($webformSubmissionData)) {
      $form_state->setErrorByName('', 'An error occurred while creating the account. Please try again later.');
      return;
    }

    if (!$this->currentUser->isAnonymous()) {
      return;
    }

    try {
      $account = $this->userManagement->createUser($webformSubmissionData);
    }
    catch (InvalidUserException $exception) {
      $form_state->setErrorByName('', 'Some fields are invalid. Please check your input and try again.');
      $this->exceptionLogger->error($exception);
      return;
    }
    catch (UserExistsException $exception) {
      $form_state->setErrorByName('', $this->t('A user with this email address or username already exists. Please use a different email or username.'));
      $this->exceptionLogger->error($exception);
      return;
    }

    $violations = $this->userManagement->getViolations($account);
    if (count($violations) > 0) {
      $userFieldMapping = $this->configuration['create_user']['mapping']['user_field_mapping'];

      foreach ($violations as $violation) {
        $userFieldName = explode('.', $violation->getPropertyPath(), 2)[0];
        $webformElementName = array_flip($userFieldMapping)[$userFieldName] ?? NULL;

        if ($webformElementName) {
          $form_state->setErrorByName($webformElementName, $violation->getMessage());
        }
        else {
          $form_state->setErrorByName('', $violation->getMessage());
        }
      }
    }

    $this->account = $account;
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(WebformSubmissionInterface $webform_submission, $update = TRUE) {
    $webformData = $this->getWebformUserData($webform_submission);
    if (!$this->currentUser->isAnonymous()) {
      $this->userManagement->updateUser($webformData, $this->currentUser->id());
    }

    $account = $this->account;

    if ($account) {
      $result = $this->userManagement->saveUser($account);

      if ($result === SAVED_NEW) {
        if ($this->configuration['create_user']['sync_with_civicrm']) {
          ($this->civicrmManagement)()?->synchronizeUfMatch($account);
        }

        $this->messenger()->addStatus($this->configuration['create_user']['email_verification_message']);
        _user_mail_notify('register_no_approval_required', $account);
        $webform_submission->setOwner($account);
        $webform_submission->save();
      }
    }
  }

  /**
   * Retrieves user data from a webform submission.
   *
   * This method processes the provided webform submission to extract user
   * data based on the defined field mapping and selected roles. It constructs
   * an array of user data that can be used for user creation or updates.
   */
  protected function getWebformUserData(WebformSubmissionInterface $webformSubmission): array {
    $webformData = $webformSubmission->getData();
    $userFieldMapping = $this->configuration['create_user']['mapping']['user_field_mapping'];
    $selectedRoles = $this->configuration['create_user']['selected_roles'];

    $userFieldData = [];

    foreach ($userFieldMapping as $userField => $webformKey) {
      $userFieldData[$userField] = $webformData[$webformKey];
    }
    $userFieldData['roles'] = array_keys(array_filter($selectedRoles, fn($value) => $value !== 0));

    return $userFieldData;
  }

}
