<?php

namespace Drupal\webform_registration_handler;

use Drupal\Core\Entity\EntityConstraintViolationListInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\user\RoleInterface;
use Drupal\user\UserInterface;
use Drupal\webform_registration_handler\Exception\InvalidUserException;
use Drupal\webform_registration_handler\Exception\UserExistsException;
use Drupal\webform_registration_handler\PluginManager\WebformUserRegistrationFieldBase;
use Drupal\webform_registration_handler\PluginManager\WebformUserRegistrationFieldProvider;

/**
 * Manages user-related operations for creating and validating Drupal user.
 *
 * @ingroup webform_registration_handler
 */
class UserManagement implements UserManagementInterface {

  /**
   * The user entity storage.
   *
   * @var \Drupal\user\UserStorageInterface
   */
  protected EntityStorageInterface $userStorage;

  /**
   * The user entity storage.
   *
   * @var \Drupal\user\RoleStorageInterface
   */
  protected EntityStorageInterface $roleStorage;

  /**
   * The field provider service.
   *
   * @var \Drupal\webform_registration_handler\PluginManager\WebformUserRegistrationFieldProvider
   */
  protected WebformUserRegistrationFieldProvider $webformUserRegistrationFieldProvider;

  /**
   * Messenger.
   */
  protected MessengerInterface $messenger;

  /**
   * Constructs a UserManagement object.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, WebformUserRegistrationFieldProvider $webformUserRegistrationFieldProvider, MessengerInterface $messenger) {
    $this->userStorage = $entityTypeManager->getStorage('user');
    $this->roleStorage = $entityTypeManager->getStorage('user_role');
    $this->messenger = $messenger;
    $this->webformUserRegistrationFieldProvider = $webformUserRegistrationFieldProvider;
  }

  /**
   * {@inheritdoc}
   */
  public function createUserData(array $userData): array {
    $userProperties = [];

    foreach ($userData as $key => $value) {
      $plugin = $this->webformUserRegistrationFieldProvider->getFieldPlugin($key);

      if (empty($plugin)) {
        $userProperties[$key] = $value;
        continue;
      }

      if (!$this->isFieldValid($plugin, $value)) {
        throw new InvalidUserException();
      }

      $userProperties[$plugin->getFieldMachineName()] = $plugin->getProcessedValue($value);
    }

    return $userProperties;
  }

  /**
   * {@inheritdoc}
   */
  public function updateUser(array $webformSubmissionData, int $userId): void {
    $user = $this->userStorage->load($userId);
    foreach ($webformSubmissionData as $key => $value) {
      $plugin = $this->webformUserRegistrationFieldProvider->getFieldPlugin($key);

      if (empty($plugin)) {
        continue;
      }

      if (!$plugin->isFieldUpdatable()) {
        continue;
      }

      $newValue = $plugin->getProcessedValue($value);

      if ($newValue === $user->get($plugin->getFieldMachineName())->getString()) {
        continue;
      }

      if ($plugin->getFieldMachineName() === 'mail') {
        $userEmail = $this->userStorage->loadByProperties(['mail' => $newValue]);

        if (!empty($userEmail)) {
          $this->messenger->addError('An account with this email address already exists.');
          continue;
        }
      }

      $user->set($plugin->getFieldMachineName(), $newValue);
    }

    $user->save();
    $this->messenger->addStatus('Your profile has been updated.');
  }

  /**
   * {@inheritdoc}
   */
  public function createUser(array $webformSubmissionData): UserInterface {
    $userData = $this->createUserData($webformSubmissionData);

    if ($this->userExists($userData)) {
      throw new UserExistsException();
    }

    /** @var \Drupal\user\UserInterface $userEntity */
    $userEntity = $this->userStorage->create($userData);

    return $userEntity;
  }

  /**
   * {@inheritdoc}
   */
  public function saveUser(UserInterface $user): int {
    $user->activate();
    return $user->save();
  }

  /**
   * {@inheritdoc}
   */
  public function getViolations(UserInterface $user): EntityConstraintViolationListInterface {
    return $user->validate();
  }

  /**
   * {@inheritdoc}
   */
  public function getListOfRoleNames(): array {
    $roles = $this->roleStorage->loadMultiple();
    unset($roles[RoleInterface::ANONYMOUS_ID]);
    return $roles;
  }

  /**
   * Validates user data to check if it is valid for user creation.
   */
  private function isFieldValid(WebformUserRegistrationFieldBase $plugin, $value): bool {

    if ($plugin->isRequired() && empty($value)) {
      return FALSE;
    }

    if (!$plugin->isValueValid($value)) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Checks if a user with the same email or username already exists.
   */
  private function userExists(array $userData): bool {
    $userEmail = $this->userStorage->loadByProperties(['mail' => $userData['mail']]);
    if (!empty($userEmail)) {
      return TRUE;
    }

    $username = $this->userStorage->loadByProperties(['name' => $userData['name']]);
    if (!empty($username)) {
      return TRUE;
    }

    return FALSE;
  }

}
