<?php

namespace Drupal\webform_registration_handler;

use Drupal\Core\Entity\EntityConstraintViolationListInterface;
use Drupal\user\UserInterface;

/**
 * Defines an interface for managing user operations.
 *
 * This interface provides methods for creating and managing user-related
 * operations.
 *
 * @ingroup webform_registration_handler
 */
interface UserManagementInterface {

  /**
   * Creates user data from the provided input.
   *
   * This method accepts an array of user data and processes it to
   * create a formatted array for user creation.
   *
   * @throws \Drupal\webform_registration_handler\Exception\InvalidUserException
   */
  public function createUserData(array $userData): array;

  /**
   * Updates an existing user with data from a webform submission.
   *
   * This method updates the specified user identified by user ID with
   * the provided webform submission data.
   */
  public function updateUser(array $webformSubmissionData, int $userId): void;

  /**
   * Creates a new user from webform submission data.
   *
   * This method creates a new user using the provided webform submission data
   * and returns the created user object.
   *
   * @throws \Drupal\webform_registration_handler\Exception\UserExistsException
   * @throws \Drupal\webform_registration_handler\Exception\InvalidUserException
   */
  public function createUser(array $webformSubmissionData): UserInterface;

  /**
   * Saves a user entity.
   *
   * This method saves the provided user entity and returns the user ID.
   */
  public function saveUser(UserInterface $user): int;

  /**
   * Gets validation violations for a user entity.
   *
   * This method returns a list of validation violations for the specified
   * user entity.
   */
  public function getViolations(UserInterface $user): EntityConstraintViolationListInterface;

  /**
   * Gets a list of role names available in the system.
   *
   * This method returns an array of role names that can be assigned to users.
   */
  public function getListOfRoleNames(): array;

}
