<?php

declare(strict_types=1);

namespace Drupal\Tests\webform_revision_ui\Functional;

use Drupal\config_revision\Entity\ConfigRevision;
use Drupal\Tests\BrowserTestBase;
use Drupal\webform\Entity\Webform;

/**
 * Tests webform revision access permissions.
 *
 * @group webform_revision_ui
 */
class WebformRevisionUiAccessTest extends BrowserTestBase {

  /**
   * Test webform.
   */
  protected Webform $webform;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'user',
    'webform',
    'webform_ui',
    'config_revision',
    'webform_revision_ui',
    'system',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Configure revision settings.
    $config = $this->config('config_revision.settings');
    $config->set('enabled_entity_types', ['webform']);
    $config->save();

    // Create revision type for webform.
    $this->container->get('entity_type.manager')
      ->getStorage('config_revision_type')
      ->create([
        'id' => 'webform',
        'label' => 'Webform',
        'description' => '',
      ])
      ->save();

    // Create test webform.
    $this->webform = $this->createTestWebform();
  }

  /**
   * Tests webform revision access permissions.
   */
  public function testWebformRevisionAccess(): void {
    // Create users with different permissions.
    $view_user = $this->drupalCreateUser(['view all webform revisions']);
    $revert_user = $this->drupalCreateUser(['revert all webform revisions']);
    $delete_user = $this->drupalCreateUser(['delete all webform revisions']);

    // Load the first revision.
    $first_revision = ConfigRevision::loadConfigRevisionByConfigId('test_webform');
    $this->assertNotNull($first_revision, 'First revision was created');

    // Test view permission.
    $this->assertTrue($first_revision->access('view', $view_user), 'User with view permission can view revision');
    $this->assertTrue($first_revision->access('view revision', $view_user), 'User with view permission can view specific revision');
    $this->assertTrue($first_revision->access('view all revisions', $view_user), 'User with view permission can view all revisions');
    // Test revert and delete permissions.
    $this->assertFalse($first_revision->access('revert', $view_user), 'User with view permission cannot revert revision');
    $this->assertFalse($first_revision->access('delete revision', $view_user), 'User with view permission cannot delete revision');

    // Create a second revision.
    $this->createWebformRevision();

    // Load current revisions.
    $current_revision = ConfigRevision::loadConfigRevisionByConfigId('test_webform');
    // Set the first revision as non-default.
    $first_revision->isDefaultRevision(FALSE);

    // Test revert permission.
    $this->assertTrue($first_revision->access('revert', $revert_user), 'User with revert permission can revert non-default revision');
    $this->assertFalse($first_revision->access('delete revision', $revert_user), 'User with revert permission cannot delete revision');
    $this->assertFalse($current_revision->access('revert', $revert_user), 'Cannot revert current revision');
    $this->assertFalse($current_revision->access('delete revision', $revert_user), 'User with revert permission cannot delete current revision');

    // Test delete permission.
    $this->assertTrue($first_revision->access('delete revision', $delete_user), 'User with delete permission can delete non-default revision');
    $this->assertFalse($first_revision->access('revert', $delete_user), 'User with delete permission cannot revert revision');
    $this->assertFalse($current_revision->access('delete revision', $delete_user), 'Cannot delete current revision');
    $this->assertFalse($current_revision->access('revert', $delete_user), 'User with delete permission cannot revert current revision');
  }

  /**
   * Tests webform revision UI access.
   */
  public function testWebformRevisionUiAccess(): void {
    $assert_session = $this->assertSession();
    $revision_url = 'admin/structure/webform/manage/test_webform/revisions';

    // Create a revision.
    $this->createWebformRevision();

    // Test UI access as webform editor.
    $webform_editor = $this->drupalCreateUser([
      'edit any webform',
      'view all webform revisions',
    ]);
    $this->drupalLogin($webform_editor);
    $this->drupalGet($revision_url);
    $assert_session->statusCodeEquals(200);
    $assert_session->pageTextContains('Revisions');
    $assert_session->pageTextNotContains('Revert');
    $assert_session->pageTextNotContains('Delete');

    // Test UI access as webform admin.
    $webform_admin = $this->drupalCreateUser([
      'edit any webform',
      'view all webform revisions',
      'revert all webform revisions',
    ]);
    $this->drupalLogin($webform_admin);
    $this->drupalGet($revision_url);
    $assert_session->statusCodeEquals(200);
    $assert_session->pageTextContains('Revisions');
    $assert_session->pageTextContains('Revert');
    $assert_session->pageTextNotContains('Delete');

    // Test UI access as webform super admin.
    $webform_super_admin = $this->drupalCreateUser([
      'edit any webform',
      'view all webform revisions',
      'revert all webform revisions',
      'delete all webform revisions',
    ]);
    $this->drupalLogin($webform_super_admin);
    $this->drupalGet($revision_url);
    $assert_session->statusCodeEquals(200);
    $assert_session->pageTextContains('Revisions');
    $assert_session->pageTextContains('Revert');
    $assert_session->pageTextContains('Delete');
  }

  /**
   * Creates a test webform.
   */
  protected function createTestWebform(): Webform {
    $webform = Webform::create([
      'id' => 'test_webform',
      'title' => 'A test form',
    ]);
    $webform->setStatus(TRUE);
    $webform->save();

    return $webform;
  }

  /**
   * Creates a revision with root textfield element.
   */
  protected function createWebformRevision(): void {
    $elements = [
      'root' => [
        '#type' => 'textfield',
        '#title' => 'root',
      ],
    ];
    $this->webform->setElements($elements);
    $this->webform->save();
  }

}
