<?php

declare(strict_types=1);

namespace Drupal\Tests\webform_revision_ui\Functional;

use Drupal\config_revision\Entity\ConfigRevision;
use Drupal\config_revision\Entity\ConfigRevisionType;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;
use Symfony\Component\HttpFoundation\Response;

/**
 * Tests webform revision entity workflow.
 *
 * @group webform_revision_ui
 * @group legacy
 */
class WebformRevisionUiTest extends BrowserTestBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'webform_revision_ui',
    'config_revision',
    'block',
  ];

  /**
   * A user who can edit a webform.
   *
   * @var \Drupal\user\Entity\User
   */
  protected $editorUser;

  /**
   * Permissions to grant editor user.
   *
   * @var array
   */
  protected $editorPermissions = [
    'edit any webform',
    'view all webform revisions',
    'revert all webform revisions',
    'delete all webform revisions',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Sets the test up.
   *
   * Tasks:
   * - Creates an editor user.
   * - Activates Webform revisioning.
   * - Places relevant blocks.
   */
  protected function setUp(): void {

    parent::setUp();

    $this->editorUser = $this->drupalCreateUser($this->editorPermissions);

    // Enable webform revision.
    $config = $this->config('config_revision.settings');
    $config->set('enabled_entity_types', ['webform']);
    $config->save();
    $webform_config_revision_type = ConfigRevisionType::create([
      'id' => 'webform',
      'label' => 'Webform',
      'description' => '',
    ]);
    $webform_config_revision_type->save();

    $this->drupalPlaceBlock('page_title_block');
    $this->drupalPlaceBlock('local_actions_block');
    $this->drupalPlaceBlock('local_tasks_block');
  }

  /**
   * Tests Config revision management for Webform.
   */
  public function testConfigRevisionWebform() {

    $assert_session = $this->assertSession();
    // Create a webform.
    $webform_storage = \Drupal::entityTypeManager()->getStorage('webform');
    $elements = [
      'root' => [
        '#type' => 'textfield',
        '#title' => 'root',
      ],
      'container' => [
        '#type' => 'container',
        '#title' => 'container',
        'child' => [
          '#type' => 'textfield',
          '#title' => 'child',
          '#default_value' => '{default value}',
        ],
      ],
    ];
    /** @var \Drupal\webform\WebformInterface $webform */
    $webform = $webform_storage->create([
      'id' => 'webform_test',
      'title' => 'Webform Test',
      'description' => 'Test webform',
      'category' => 'Test',
    ]);
    $webform->setElements($elements);
    $webform->setStatus(TRUE);
    $webform->save();
    $webform = $webform_storage->loadUnchanged('webform_test');

    // Verify the webform is created properly.
    $this->drupalGet($webform->toUrl());
    $assert_session->statusCodeEquals(Response::HTTP_OK);
    $assert_session->fieldExists('root');
    $assert_session->fieldExists('child');

    // Verify the webform revision has been saved properly.
    $webform_revision = ConfigRevision::loadConfigRevisionByConfigId('webform_test');
    $this->assertNotNull($webform_revision);

    // Verify the webform revision can be viewed by editors.
    $this->drupalLogin($this->editorUser);
    $this->drupalGet($webform_revision->toUrl());
    $assert_session->statusCodeEquals(Response::HTTP_OK);

    // Save a new webform revision.
    $webform->deleteElement('container');
    $webform->save();

    // Verify the webform is updated properly.
    $this->drupalGet($webform->toUrl());
    $assert_session->statusCodeEquals(Response::HTTP_OK);
    $assert_session->fieldExists('root');
    $assert_session->fieldNotExists('child');

    $webform_id = $webform->id();
    // Verify the webform revision history page is showing all the revisions.
    $this->drupalGet(Url::fromRoute('entity.webform.version-history', ['webform' => $webform_id]));

    $assert_session->statusCodeEquals(Response::HTTP_OK);
    $this->assertSession()->elementsCount('css', 'table tbody tr', 2);
    // Current revision text is found on the latest revision row.
    $this->assertSession()->elementTextContains('css', 'table tbody tr:nth-child(1)', 'Current revision');

    $date = \Drupal::service('date.formatter')->format($webform_revision->getRevisionCreationTime());

    // Verify the webform revision revert page is showing the diff table.
    $this->drupalGet(Url::fromRoute('entity.config_revision.revision_revert_form', [
      'config_revision' => $webform_revision->id(),
      'config_revision_revision' => $webform_revision->getRevisionId(),
    ]));
    $assert_session->statusCodeEquals(Response::HTTP_OK);
    $assert_session->responseContains($this->t('Are you sure you want to revert to the revision from %revision-date?', [
      '%revision-date' => $date,
    ]));
    $assert_session->elementExists('css', 'table.diff');

    // Revert back to the first revision.
    $this->submitForm([], 'Revert');
    $assert_session->statusCodeEquals(Response::HTTP_OK);
    $assert_session->responseContains($this->t('@type %title has been reverted to the revision from %revision-date.', [
      '@type' => 'Webform',
      '%title' => $webform_revision->label(),
      '%revision-date' => $date,
    ]));
    $this->assertSession()->elementsCount('css', 'table tbody tr', 3);
    // Current revision text is found on the latest revision row.
    $this->assertSession()->elementTextContains('css', 'table tbody tr:nth-child(1)', 'Current revision');

    // Verify the webform is updated properly.
    $this->drupalGet($webform->toUrl());
    $assert_session->statusCodeEquals(Response::HTTP_OK);
    $assert_session->fieldExists('root');
    $assert_session->fieldExists('child');
    $webform = $webform_storage->loadUnchanged('webform_test');
    // Check elements decoded and flattened.
    $flattened_elements = [
      'root' => [
        '#type' => 'textfield',
        '#title' => 'root',
      ],
      'container' => [
        '#type' => 'container',
        '#title' => 'container',
      ],
      'child' => [
        '#type' => 'textfield',
        '#title' => 'child',
        '#default_value' => '{default value}',
      ],
    ];
    $this->assertEquals($webform->getElementsDecodedAndFlattened(), $flattened_elements);

    // Delete the first revision.
    $this->drupalGet(Url::fromRoute('entity.config_revision.revision_delete_form', [
      'config_revision' => $webform_revision->id(),
      'config_revision_revision' => $webform_revision->getRevisionId(),
    ]));
    $assert_session->statusCodeEquals(Response::HTTP_OK);
    $assert_session->responseContains($this->t('Are you sure you want to delete the revision from %revision-date?', [
      '%revision-date' => $date,
    ]));
    $this->submitForm([], 'Delete');
    $assert_session->statusCodeEquals(Response::HTTP_OK);
    $assert_session->responseContains($this->t('Revision from %revision-date of @type %title has been deleted.', [
      '@type' => 'Webform',
      '%title' => $webform_revision->label(),
      '%revision-date' => $date,
    ]));
    $this->assertSession()->elementsCount('css', 'table tbody tr', 2);
    // Current revision text is found on the latest revision row.
    $this->assertSession()->elementTextContains('css', 'table tbody tr:nth-child(1)', 'Current revision');
  }

  /**
   * Tests presence of the "Revisions" tab.
   *
   * Form editors should see the "Revisions" tab in Webform pages.
   */
  public function testRevisionsTabPresence(): void {

    $test_webform = \Drupal::entityTypeManager()->getStorage('webform')->create([
      'id'    => 'test-form',
      'title' => 'A test form',
    ]);
    $test_webform->save();
    $test_webform_rev_url = Url::fromRoute('entity.webform.version-history', ['webform' => $test_webform->id()]);

    $assert_session = $this->assertSession();

    // Anonymous user should not see the "Revisions" tab.
    $this->drupalGet($test_webform->toUrl());
    $assert_session->statusCodeEquals(Response::HTTP_OK);
    $assert_session->linkNotExistsExact('Revisions');
    $assert_session->linkByHrefNotExists($test_webform_rev_url->toString());

    // Editor user should see the "Revisions" tab.
    $this->drupalLogin($this->editorUser);
    $this->drupalGet($test_webform->toUrl());
    $assert_session->statusCodeEquals(Response::HTTP_OK);
    $assert_session->linkExistsExact('Revisions');
    $assert_session->linkByHrefExists($test_webform_rev_url->toString());
  }

}
