<?php

namespace Drupal\webform_simplify\Element;

use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\Element\WebformRoles;

/**
 * Provides a webform roles (checkboxes) element.
 *
 * It extends from the webform_roles checkboxes element, but intentionally
 * disables pre-selected fields from webform_simplify.
 *
 * @FormElement("webform_simplify_roles")
 */
class WebformSimplifyRoles extends WebformRoles {

  /**
   * {@inheritdoc}
   */
  public static function processCheckboxes(&$element, FormStateInterface $form_state, &$complete_form) {
    $element = parent::processCheckboxes($element, $form_state, $complete_form);

    // Filter out the roles that are not enabled.
    $disabledRoles = \Drupal::config('webform_simplify.settings')->get('webform_settings.disabled_access_roles_settings') ?? [];
    $options = $element['#options'];

    foreach ($disabledRoles as $disabledRole) {
      if (array_key_exists($disabledRole, $options)) {
        // Only disable the singular checkbox field, as we still want the value
        // to be saved, avoiding overwriting whatever someone with bypass
        // webform_simplify permissions has set.
        $element[$disabledRole]['#disabled'] = TRUE;
      }
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function validateWebformRoles(&$element, FormStateInterface $form_state, &$complete_form) {
    parent::validateWebformRoles($element, $form_state, $complete_form);

    // Ensure that pre-selected elements that were disabled still get included
    // in the submission, otherwise we might lose their values unintentionally.
    $disabledRoles = \Drupal::config('webform_simplify.settings')->get('webform_settings.disabled_access_roles_settings') ?? [];
    foreach ($disabledRoles as $disabledRole) {
      if (!array_key_exists($disabledRole, $element)) {
        continue;
      }

      // Get the default value for this role (i.e. The value it was before
      // saving).
      $found = array_search($disabledRole, $element['#default_value']);
      if ($found === FALSE) {
        continue;
      }

      // If it was selected, add it to the currently selected values as well, so
      // it gets submitted (disabled fields are not included in the HTTP
      // submission by default).
      if (!array_key_exists('#value', $element)) {
        $element['#value'] = [];
      }
      $element['#value'] = array_merge($element['#value'], [$disabledRole => $disabledRole]);
    }

    $form_state->setValueForElement($element, $element['#value']);
  }

}
