<?php

namespace Drupal\webform_statistics\Plugin\views\field;

use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Field handler to display submission language name.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("webform_submission_language")
 */
class WebformSubmissionLanguage extends FieldPluginBase {

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * Constructs a WebformSubmissionLanguage object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, LanguageManagerInterface $language_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('language_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    $this->ensureMyTable();
    $this->field_alias = $this->query->addField($this->tableAlias, 'langcode', 'webform_submission_language_langcode');
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    $langcode = $this->getValue($values);

    if (empty($langcode)) {
      return '';
    }

    // Get the language name.
    $language = $this->languageManager->getLanguage($langcode);

    return $language ? $language->getName() : $langcode;
  }

}
