(function (Drupal, once) {
  /**
   * Calculate date range based on time range option.
   *
   * @param {string} range - The time range option (24h, 7d, 30d, etc).
   * @return {Object|null} Object with from and to dates, or null if invalid.
   */
  function getDateRange(range) {
    const today = new Date();
    const toDate = today.toISOString().split('T')[0];
    let fromDate;

    switch (range) {
      case '24h':
        fromDate = new Date(today.getTime() - 24 * 60 * 60 * 1000);
        break;
      case '7d':
        fromDate = new Date(today.getTime() - 7 * 24 * 60 * 60 * 1000);
        break;
      case '30d':
        fromDate = new Date(today.getTime() - 30 * 24 * 60 * 60 * 1000);
        break;
      case '90d':
        fromDate = new Date(today.getTime() - 90 * 24 * 60 * 60 * 1000);
        break;
      case '180d':
        fromDate = new Date(today.getTime() - 180 * 24 * 60 * 60 * 1000);
        break;
      case '365d':
        fromDate = new Date(today.getTime() - 365 * 24 * 60 * 60 * 1000);
        break;
      default:
        return null;
    }

    return {
      from: fromDate.toISOString().split('T')[0],
      to: toDate,
    };
  }

  /**
   * Update wrapper class based on preset selection.
   *
   * @param {HTMLElement} wrapper - The wrapper element.
   * @param {boolean} isPreset - Whether a preset is selected.
   */
  function updateWrapperState(wrapper, isPreset) {
    if (isPreset) {
      wrapper.classList.add('preset-selected');
    } else {
      wrapper.classList.remove('preset-selected');
    }
  }

  /**
   * Time range selector behavior.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.webformStatisticsTimeRange = {
    attach(context) {
      const timeRangeSelects = once(
        'webform-statistics-time-range',
        '.webform-statistics-time-range',
        context,
      );

      timeRangeSelects.forEach(function (select) {
        const form = select.closest('form');
        if (!form) {
          return;
        }

        const dateFromWrapper = form.querySelector(
          '.webform-statistics-date-from',
        );
        const dateToWrapper = form.querySelector('.webform-statistics-date-to');
        const dateFromInput = form.querySelector('[name="created_from"]');
        const dateToInput = form.querySelector('[name="created_to"]');

        if (
          !dateFromInput ||
          !dateToInput ||
          !dateFromWrapper ||
          !dateToWrapper
        ) {
          return;
        }

        // Reorganize DOM: wrap time range and dates together.
        const selectWrapper = select.closest('.form-item');
        if (
          selectWrapper &&
          !selectWrapper.parentElement.classList.contains(
            'webform-statistics-time-range-wrapper',
          )
        ) {
          const wrapper = document.createElement('div');
          wrapper.className = 'webform-statistics-time-range-wrapper';

          const dateFields = document.createElement('div');
          dateFields.className = 'webform-statistics-date-fields';

          selectWrapper.parentNode.insertBefore(wrapper, selectWrapper);
          wrapper.appendChild(selectWrapper);
          wrapper.appendChild(dateFields);
          dateFields.appendChild(dateFromWrapper);
          dateFields.appendChild(dateToWrapper);

          // Set initial state.
          updateWrapperState(wrapper, select.value !== '');
        }

        const wrapper = select.closest(
          '.webform-statistics-time-range-wrapper',
        );

        // Update dates when time range changes.
        select.addEventListener('change', function () {
          const range = this.value;

          if (range) {
            const dates = getDateRange(range);
            if (dates) {
              dateFromInput.value = dates.from;
              dateToInput.value = dates.to;
            }
          }

          if (wrapper) {
            updateWrapperState(wrapper, range !== '');
          }
        });

        // Set time range to custom when dates are manually changed.
        dateFromInput.addEventListener('change', function () {
          select.value = '';
          if (wrapper) {
            updateWrapperState(wrapper, false);
          }
        });

        dateToInput.addEventListener('change', function () {
          select.value = '';
          if (wrapper) {
            updateWrapperState(wrapper, false);
          }
        });
      });
    },
  };
})(Drupal, once);
