<?php

namespace Drupal\webform_statistics\Plugin\views\field;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Url;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Field handler to display webform label with optional language suffix.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("webform_submission_label")
 */
class WebformSubmissionLabel extends FieldPluginBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * Constructs a WebformSubmissionLabel object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, LanguageManagerInterface $language_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('language_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['link_to_webform'] = ['default' => TRUE];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    $form['link_to_webform'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Link to webform'),
      '#description' => $this->t('Make the webform title a link to the webform results page.'),
      '#default_value' => $this->options['link_to_webform'],
    ];
    parent::buildOptionsForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    $this->ensureMyTable();
    // Add webform_id field.
    $this->field_alias = $this->query->addField($this->tableAlias, 'webform_id', 'webform_submission_label_webform_id');
    // Add langcode field.
    $this->query->addField($this->tableAlias, 'langcode', 'webform_submission_label_langcode');
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    $webform_id = $this->getValue($values);
    $submission_langcode = $values->webform_submission_label_langcode ?? NULL;

    if (empty($webform_id)) {
      return '';
    }

    // Load the webform entity.
    $webform = $this->entityTypeManager->getStorage('webform')->load($webform_id);
    if (!$webform) {
      return $webform_id;
    }

    // Get webform title.
    $title = $webform->label();

    // Return with or without link based on option.
    if ($this->options['link_to_webform']) {
      $url = Url::fromRoute('entity.webform.results_submissions', ['webform' => $webform_id]);
      return [
        '#type' => 'link',
        '#title' => $title,
        '#url' => $url,
      ];
    }

    // For charts (no link), always include language suffix.
    if ($submission_langcode) {
      $language = $this->languageManager->getLanguage($submission_langcode);
      $language_name = $language ? $language->getName() : $submission_langcode;
      $title .= ' (' . $language_name . ')';
    }

    return $title;
  }

}
