<?php

namespace Drupal\webform_yuboto\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\webform_yuboto\YubotoApiService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Yuboto SMS settings for this site.
 */
class YubotoSettingsForm extends ConfigFormBase {

  /**
   * The Yuboto API service.
   *
   * @var \Drupal\webform_yuboto\YubotoApiService
   */
  protected YubotoApiService $yubotoApi;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    $instance = parent::create($container);
    $instance->yubotoApi = $container->get('webform_yuboto.api_service');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'webform_yuboto_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['webform_yuboto.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('webform_yuboto.settings');

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable SMS sending'),
      '#default_value' => $config->get('enabled'),
      '#description' => $this->t('Check this box to enable SMS sending functionality. Uncheck to disable without uninstalling the module. Useful for staging environments.'),
      '#weight' => -10,
    ];

    $form['debug_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable debug mode'),
      '#default_value' => $config->get('debug_enabled'),
      '#description' => $this->t('When enabled, exceptions are thrown for errors which will interrupt form submission. When disabled, errors are logged but don\'t interrupt the user experience.'),
      '#weight' => -5,
    ];

    $form['settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('SMS Settings'),
      '#states' => [
        'visible' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
        'disabled' => [
          ':input[name="enabled"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['settings']['yuboto_api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Yuboto API Key'),
      '#default_value' => $config->get('yuboto_api_key'),
      '#description' => $this->t('Enter your Yuboto API key. You can get this from your Yuboto account dashboard.'),
      '#required' => TRUE,
      '#parents' => ['yuboto_api_key'],
    ];

    $form['settings']['default_sender'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default Sender Name'),
      '#default_value' => $config->get('default_sender'),
      '#description' => $this->t('Default sender name for SMS messages. You can use tokens like [site:name].'),
      '#placeholder' => '[site:name]',
      '#parents' => ['default_sender'],
    ];

    $form['settings']['default_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Default Message'),
      '#default_value' => $config->get('default_message'),
      '#description' => $this->t('Default SMS message. You can use tokens from the webform submission and global Drupal tokens.'),
      '#rows' => 3,
      '#parents' => ['default_message'],
    ];

    $form['settings']['token_help'] = [
      '#theme' => 'token_tree_link',
      '#token_types' => ['webform', 'webform_submission', 'site'],
      '#global_types' => TRUE,
      '#click_insert' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $api_key = $form_state->getValue('yuboto_api_key');
    $enabled = $form_state->getValue('enabled');

    // Only validate API key if SMS sending is enabled
    if ($enabled && !empty($api_key)) {
      // Temporarily set the API key in the service to validate it
      $this->yubotoApi->setApiKeyForValidation($api_key);

      if (!$this->yubotoApi->validateApiKey()) {
        $form_state->setErrorByName('yuboto_api_key', $this->t('The Yuboto API key is invalid. Please check your API key and try again.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('webform_yuboto.settings')
      ->set('enabled', $form_state->getValue('enabled'))
      ->set('debug_enabled', $form_state->getValue('debug_enabled'))
      ->set('yuboto_api_key', $form_state->getValue('yuboto_api_key'))
      ->set('default_sender', $form_state->getValue('default_sender'))
      ->set('default_message', $form_state->getValue('default_message'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
