<?php

namespace Drupal\webform_yuboto;

use Drupal\backup_migrate\Core\Translation\TranslatableTrait;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Service for interacting with the Yuboto SMS API.
 */
class YubotoApiService {

  use TranslatableTrait;

  /**
   * The Yuboto API endpoint.
   */
  protected const string API_ENDPOINT = 'https://services.yuboto.com/omni/v1/Send';

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * The logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * Temporary API key for validation.
   *
   * @var string|null
   */
  private ?string $validationApiKey = null;

  /**
   * Constructs a YubotoApiService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory, ClientInterface $http_client) {
    $this->configFactory = $config_factory;
    $this->logger = $logger_factory->get('webform_yuboto');
    $this->httpClient = $http_client;
  }

  /**
   * Send an SMS message using the Yuboto API.
   *
   * @param string $phone_number
   *   The recipient phone number.
   * @param string|null $message
   *   The SMS message content.
   * @param string|null $sender
   *   The sender name (optional).
   * @param array $additional_options
   *   Additional SMS options (optional).
   *
   * @return array
   *   The API response.
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function sendSms(string $phone_number, ?string $message, ?string $sender, array $additional_options = []): array {
    $config = $this->configFactory->get('webform_yuboto.settings');

    // Check if SMS sending is enabled
    if (!$config->get('enabled')) {
      $this->logger->info('SMS sending is disabled. SMS to @phone was not sent.', [
        '@phone' => $phone_number,
      ]);
      return ['status' => 'disabled', 'message' => 'Yuboto SMS sending is disabled'];
    }

    $api_key = $config->get('yuboto_api_key');

    if (empty($api_key)) {
      $error_message = 'Yuboto API key is not configured.';
      if ($config->get('debug_enabled')) {
        $this->logger->error($error_message);
      }
      return ['status' => 'error', 'message' => $error_message];
    }

    // Validate and normalize the phone number
    $phone_validation = $this->validateGreekMobileNumber($phone_number);

    if (!$phone_validation['valid']) {
      $error_message = 'Invalid phone number: ' . $phone_validation['error'];
      if ($config->get('debug_enabled')) {
        $this->logger->error($error_message);
      }
      return ['status' => 'error', 'message' => $error_message];
    }

    // Use the validated and normalized phone number
    $phone_number = $phone_validation['normalized'];

    // Prepare SMS data according to v1.10 API specification.
    $sms_data = [
      "dir" => "false",
      "contacts" => [
        [
          "phonenumber" => $phone_number
        ]
      ],
      "sms" => [
        "sender" => $sender,
        "text" => $message,
        "validity" => $additional_options['validity'] ?? 1440,
        "typesms" => "sms",
        "long_sms" => $additional_options['longsms'] ?? false,
        "priority" => $additional_options['priority'] ?? 1,
      ],
    ];

    // Allow other modules to alter the SMS data before sending.
    $context = [
      'phone_number' => $phone_number,
      'message' => $message,
      'sender' => $sender,
      'additional_options' => $additional_options,
    ];

    // Dispatch hook to allow alteration of SMS data.
    \Drupal::moduleHandler()->alter('webform_yuboto_sms_data', $sms_data, $context);

    if ($this->isDebugEnabled()) {
      $this->logger->info('Preparing to send SMS to @phone: @message', [
        '@phone' => $phone_number,
        '@message' => $message,
      ]);
    }

    $response = $this->httpClient->post(self::API_ENDPOINT, [
      'headers' => [
        'Content-Type' => 'application/json; charset=utf-8',
        'Accept' => 'application/json',
        'Authorization' => 'Basic ' . base64_encode($api_key),
      ],
      'json' => $sms_data,
      'timeout' => 30,
    ]);

    $response_data = json_decode($response->getBody()->getContents(), TRUE);

    // Check if the API returned an error and exit early.
    if (isset($response_data['ErrorCode']) && $response_data['ErrorCode'] !== 0) {
      $error_code = $response_data['ErrorCode'];
      $detailed_message = $response_data['ErrorMessage'] ?? 'Unknown API error';
      $error_message = "Yuboto API error {$error_code}: {$detailed_message}";
      $this->logger->error($error_message);

      return [
        'status' => 'error',
        'message' => $error_message,
      ];
    } else {
      if ($this->isDebugEnabled()) {
        $message = $this->t("Yuboto SMS sent message successfully to @phone.", [
          '@phone' => $phone_number,
        ]);
        $this->logger->info($message);
      }

      return $response_data;
    }
  }

  /**
   * Validate the API key by making a test request.
   *
   * @return bool
   *   TRUE if the API key is valid, FALSE otherwise.
   */
  public function validateApiKey(): bool {
    try {
      // Use validation API key if set, otherwise use the configured one
      $api_key = $this->validationApiKey ?? $this->configFactory->get('webform_yuboto.settings')->get('yuboto_api_key');

      if (empty($api_key)) {
        $this->validationApiKey = null; // Clear validation key if empty
        return FALSE;
      }

      // Make a simple request to check API key validity using the balance endpoint
      try {
        $balance_endpoint = str_replace('/Send', '/UserBalance', self::API_ENDPOINT);
        $response = $this->httpClient->get($balance_endpoint, [
          'headers' => [
            'Content-Type' => 'application/json; charset=utf-8',
            'Accept' => 'application/json',
            'Authorization' => 'Basic ' . base64_encode($api_key),
          ],
          'timeout' => 10,
        ]);
        $is_valid = $response->getStatusCode() === 200;
      $this->validationApiKey = null; // Clear validation key after use
      return $is_valid;
      }
      catch (GuzzleException $e) {
        $this->logger->error('API test failed: @error', [
          '@error' => $e->getMessage(),
        ]);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('API key validation failed: @error', [
        '@error' => $e->getMessage(),
      ]);
      $this->validationApiKey = null; // Clear validation key
      return FALSE;
    }
    // If we reach this point, something went wrong so log it and return false.
    $this->logger->error('API key validation failed without exception.');
    $this->validationApiKey = null; // Clear validation key
    return FALSE;
  }

  /**
   * Get the current API key.
   *
   * @return string
   *   The configured API key.
   */
  public function getApiKey(): string {
    return $this->configFactory->get('webform_yuboto.settings')->get('yuboto_api_key') ?: '';
  }

  /**
   * Set API key temporarily for validation.
   *
   * This method allows temporarily overriding the stored API key for validation
   * purposes without persisting the change to configuration.
   *
   * @param string $api_key
   *   The API key to set for validation.
   */
  public function setApiKeyForValidation(string $api_key): void {
    $this->validationApiKey = $api_key;
  }

  /**
   * Check if SMS sending is enabled.
   *
   * @return bool
   *   TRUE if SMS sending is enabled, FALSE otherwise.
   */
  public function isEnabled(): bool {
    return (bool) $this->configFactory->get('webform_yuboto.settings')->get('enabled');
  }

  /**
   * Check if debug mode is enabled.
   *
   * @return bool
   *   TRUE if debug mode is enabled, FALSE otherwise.
   */
  public function isDebugEnabled(): bool {
    return (bool) $this->configFactory->get('webform_yuboto.settings')->get('debug_enabled');
  }

  /**
   * Validate and normalize a Greek mobile phone number.
   *
   * @param string $phone_number
   *   The phone number to validate.
   *
   * @return array
   *   An associative array with:
   *   - 'valid' (bool): Whether the number is valid
   *   - 'normalized' (string): The normalized phone number with 30 prefix (format: 3069XXXXXXXX)
   *   - 'error' (string): Error message if invalid, empty if valid
   */
  public function validateGreekMobileNumber(string $phone_number): array {
    // Remove all non-digit characters
    $cleaned = preg_replace('/\D/', '', $phone_number);

    // Handle empty input
    if (empty($cleaned)) {
      return [
        'valid' => FALSE,
        'normalized' => '',
        'error' => 'Phone number is empty',
      ];
    }

    // Handle numbers that already have country code (30 prefix)
    if (strlen($cleaned) === 12 && str_starts_with($cleaned, '30')) {
      // Check if it's a Greek mobile number (starts with 69 after country code)
      if (substr($cleaned, 2, 2) === '69') {
        return [
          'valid' => TRUE,
          'normalized' => $cleaned,
          'error' => '',
        ];
      }
      else {
        return [
          'valid' => FALSE,
          'normalized' => '',
          'error' => 'Phone number must be a Greek mobile number starting with 69',
        ];
      }
    }

    // Handle 10-digit Greek mobile numbers (add country code)
    elseif (strlen($cleaned) === 10) {
      // Check if it's a Greek mobile number (starts with 69)
      if (str_starts_with($cleaned, '69')) {
        return [
          'valid' => TRUE,
          'normalized' => '30' . $cleaned,
          'error' => '',
        ];
      }
      else {
        return [
          'valid' => FALSE,
          'normalized' => '',
          'error' => 'Phone number must be a Greek mobile number starting with 69',
        ];
      }
    }

    // Invalid length or format
    else {
      return [
        'valid' => FALSE,
        'normalized' => '',
        'error' => 'Invalid phone number format. Expected 10-digit Greek mobile number starting with 69, or 12-digit number with 30 country code',
      ];
    }
  }

}
