<?php

/**
 * @file
 * Hooks provided by the Webform Yuboto module.
 */

/**
 * @addtogroup hooks
 * @{
 */

/**
 * Allows modules to alter the SMS request data before it is sent to the Yuboto API.
 *
 * This hook is called just before the HTTP request is made to the Yuboto API,
 * allowing modules to modify any aspect of the SMS data structure, add custom
 * parameters, or implement conditional logic based on message content or
 * recipient information.
 *
 * @param array &$sms_data
 *   The SMS data array that will be sent to the Yuboto API. This is passed by
 *   reference and any modifications will affect the actual API request.
 *   The default structure follows the simplified Yuboto API format:
 *   [
 *     'dir' => 'false',
 *     'contacts' => [
 *       [
 *         'phonenumber' => '306912345678',
 *       ],
 *     ],
 *     'sms' => [
 *       'sender' => 'Sender',
 *       'text' => 'Your message here',
 *       'validity' => 1440,
 *       'typesms' => 'sms',
 *       'long_sms' => false,
 *       'priority' => 1,
 *     ],
 *   ]
 * @param array $context
 *   Context information about the SMS being sent:
 *   - phone_number (string): The normalized phone number (format: 3069XXXXXXXX)
 *   - message (string): The SMS message content
 *   - sender (string): The sender name
 *   - additional_options (array): Additional SMS options passed to the handler
 *
 * @see \Drupal\webform_yuboto\YubotoApiService::sendSms()
 *
 * @ingroup webform_yuboto_hooks
 */
function hook_webform_yuboto_sms_data(array &$sms_data, array $context) {
  // Example: Set high priority for urgent messages
  if (strpos($context['message'], 'URGENT') !== FALSE) {
    $sms_data['sms']['priority'] = 2;
  }

  // Example: Modify sender based on phone number prefix
  if (substr($context['phone_number'], 2, 2) === '98') {
    $sms_data['sms']['sender'] = 'SpecialService';
  }

  // Example: Enable long SMS for messages with special characters
  if (preg_match('/[^\x00-\x7F]/', $context['message'])) {
    $sms_data['sms']['long_sms'] = true;
  }

  // Example: Set custom validity period
  if (isset($context['additional_options']['validity'])) {
    $sms_data['sms']['validity'] = $context['additional_options']['validity'];
  }

  // Example: Set different SMS type
  if (isset($context['additional_options']['typesms'])) {
    $sms_data['sms']['typesms'] = $context['additional_options']['typesms'];
  }

  // Example: Add custom additional options
  if (isset($context['additional_options']['priority'])) {
    $sms_data['sms']['priority'] = $context['additional_options']['priority'];
  }
}

/**
 * Allows modules to skip sending SMS for specific webform submissions.
 *
 * This hook is called during the WebformYubotoHandler::postSave() method,
 * allowing modules to conditionally prevent SMS from being sent based on
 * the submission data, webform configuration, or any other criteria.
 *
 * @param bool &$skip_sms
 *   (optional) By default FALSE. Set to TRUE to skip SMS sending for this
 *   submission. This parameter is passed by reference.
 * @param \Drupal\webform\WebformSubmissionInterface $webform_submission
 *   The webform submission entity that would trigger the SMS.
 * @param \Drupal\webform_yuboto\Plugin\WebformHandler\WebformYubotoHandler $handler
 *   The webform handler instance that is processing the submission.
 *
 * @see \Drupal\webform_yuboto\Plugin\WebformHandler\WebformYubotoHandler::postSave()
 *
 * @ingroup webform_yuboto_hooks
 */
function hook_webform_yuboto_skip_sms(&$skip_sms, \Drupal\webform\WebformSubmissionInterface $webform_submission, \Drupal\webform_yuboto\Plugin\WebformHandler\WebformYubotoHandler $handler) {
  // Example: Skip SMS for specific webform ID
  if ($webform_submission->getWebform()->id() === 'test_form') {
    $skip_sms = TRUE;
  }

  // Example: Skip SMS based on submission data
  $data = $webform_submission->getData();
  if (isset($data['skip_sms']) && $data['skip_sms'] === '1') {
    $skip_sms = TRUE;
  }

  // Example: Skip SMS for specific phone number patterns
  if (isset($data['phone']) && preg_match('/^(\+30)?698/', $data['phone'])) {
    $skip_sms = TRUE;
  }

  // Example: Skip SMS based on time of day
  $current_hour = (int) date('H');
  if ($current_hour >= 22 || $current_hour <= 6) {
    $skip_sms = TRUE;
  }

  // Example: Skip SMS for specific user roles
  if ($webform_submission->getOwner() && $webform_submission->getOwner()->hasRole('administrator')) {
    $skip_sms = TRUE;
  }
}

/**
 * @} End of "addtogroup hooks".
 */