<?php

namespace Drupal\webform_yuboto\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\webform_yuboto\YubotoApiService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Admin test form to send a demo SMS using default module settings.
 */
class YubotoTestSmsForm extends FormBase {

  use StringTranslationTrait;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The Yuboto API service.
   *
   * @var \Drupal\webform_yuboto\YubotoApiService
   */
  protected YubotoApiService $yubotoApi;

  /**
   * The logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * The token service (optional).
   *
   * @var object|null
   */
  protected ?object $token;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    $instance = new static();
    $instance->configFactory = $container->get('config.factory');
    $instance->yubotoApi = $container->get('webform_yuboto.api_service');
    $instance->logger = $container->get('logger.factory')->get('webform_yuboto');

    // Token module is not guaranteed as a dependency; use if available.
    $instance->token = $container->has('token') ? $container->get('token') : NULL;

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'webform_yuboto_test_sms';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->configFactory->get('webform_yuboto.settings');

    $enabled = (bool) $config->get('enabled');
    $api_key = (string) ($config->get('yuboto_api_key') ?? '');
    $has_api_key = trim($api_key) !== '';
    $default_sender_raw = (string) ($config->get('default_sender') ?? '');
    $default_message_raw = (string) ($config->get('default_message') ?? '');
    $default_sender = $this->replaceTokens($default_sender_raw);
    $default_message = $this->replaceTokens($default_message_raw);

    $status_lines = [];
    $status_lines[] = $enabled
      ? $this->t('<strong>Status</strong>: Enabled')
      : $this->t('<strong>Status</strong>: Disabled (SMS will not be sent)');

    $status_lines[] = $has_api_key
      ? $this->t('<strong>API key</strong>: Configured')
      : $this->t('<strong>API key</strong>: Missing (configure it first)');

    $form['intro'] = [
      '#type' => 'markup',
      '#markup' => Markup::create('<p>' . $this->t('Use this form to send a single demo SMS using the default module settings.') . '</p>'),
    ];

    $form['status'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Current configuration'),
    ];

    $form['status']['summary'] = [
      '#type' => 'markup',
      '#markup' => Markup::create('<div>' . implode('<br>', array_map('strval', $status_lines)) . '</div>'),
    ];

    $form['status']['defaults'] = [
      '#type' => 'details',
      '#title' => $this->t('Defaults that will be used'),
      '#open' => TRUE,
    ];

    $form['status']['defaults']['sender'] = [
      '#type' => 'item',
      '#title' => $this->t('Sender'),
      '#markup' => $default_sender !== '' ? $default_sender : $this->t('<em>Empty</em>'),
    ];

    $form['status']['defaults']['message'] = [
      '#type' => 'item',
      '#title' => $this->t('Message'),
      '#markup' => $default_message !== '' ? nl2br($default_message) : $this->t('<em>Empty</em>'),
    ];

    $form['phone_number'] = [
      '#type' => 'tel',
      '#title' => $this->t('Recipient phone number'),
      '#required' => TRUE,
      '#description' => $this->t('Greek mobile only. Accepted formats: 6912345678 or 306912345678. The module will normalize to 3069XXXXXXXX.'),
      '#placeholder' => '6912345678',
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Send test SMS'),
      '#button_type' => 'primary',
      '#disabled' => !$enabled || !$has_api_key,
    ];

    if (!$enabled || !$has_api_key) {
      $form['actions']['help'] = [
        '#type' => 'markup',
        '#markup' => Markup::create('<p><em>' . $this->t('Sending is disabled until the module is enabled and an API key is configured.') . '</em></p>'),
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->configFactory->get('webform_yuboto.settings');
    $phone_number = (string) $form_state->getValue('phone_number');
    $sender = $this->replaceTokens((string) ($config->get('default_sender') ?? ''));
    $message = $this->replaceTokens((string) ($config->get('default_message') ?? ''));
    $result = $this->yubotoApi->sendSms($phone_number, $message, $sender);

    if (($result['status'] ?? NULL) === 'disabled') {
      $this->messenger()->addWarning($this->t('Yuboto SMS sending is currently disabled in module settings.'));
      return;
    }

    if (($result['status'] ?? NULL) === 'error') {
      $this->messenger()->addError($this->t('Failed to send test SMS: @message', [
        '@message' => (string) ($result['message'] ?? 'Unknown error'),
      ]));
      return;
    }

    // Otherwise treat it as a successful API response payload.
    $this->messenger()->addStatus($this->t('Test SMS request sent. Check your phone and logs for details.'));
    $this->logger->info('Test SMS submitted to @phone. API response: @response', [
      '@phone' => $phone_number,
      '@response' => json_encode($result, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
    ]);
  }

  /**
   * Replace tokens in a string if token service is available.
   *
   * @param string $value
   *   Raw value with tokens.
   *
   * @return string
   *   Value with tokens replaced when possible.
   */
  protected function replaceTokens(string $value): string {
    if ($value === '') {
      return '';
    }

    if ($this->token && method_exists($this->token, 'replace')) {
      try {
        // Provide minimal context: allow [site:*] and other global tokens.
        return (string) $this->token->replace($value, [], ['clear' => TRUE]);
      }
      catch (\Throwable $e) {
        // Fall back to raw value; this is a test UI so don't hard-fail.
        $this->logger->warning('Token replacement failed in test form: @error', [
          '@error' => $e->getMessage(),
        ]);
      }
    }

    return $value;
  }

}
