<?php

namespace Drupal\webform_yuboto\Plugin\WebformHandler;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\Plugin\WebformHandlerBase;
use Drupal\webform\WebformSubmissionInterface;
use Drupal\webform_yuboto\YubotoApiService;
use GuzzleHttp\Exception\GuzzleException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form submission to Yuboto SMS handler.
 *
 * @WebformHandler(
 *   id = "yuboto_sms",
 *   label = @Translation("Yuboto SMS"),
 *   category = @Translation("SMS"),
 *   description = @Translation("Sends an SMS message to the phone number from
 *   the form submission."),
 *    cardinality = \Drupal\webform\Plugin\WebformHandlerInterface::CARDINALITY_UNLIMITED,
 *   results = \Drupal\webform\Plugin\WebformHandlerInterface::RESULTS_PROCESSED,
 *   submission = \Drupal\webform\Plugin\WebformHandlerInterface::SUBMISSION_OPTIONAL,
 * )
 */
class WebformYubotoHandler extends WebformHandlerBase {

  /**
   * The Yuboto API service.
   *
   * @var \Drupal\webform_yuboto\YubotoApiService
   */
  protected YubotoApiService $yubotoApi;

  /**
   * The token manager.
   *
   * @var \Drupal\webform\WebformTokenManagerInterface
   */
  protected $tokenManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->yubotoApi = $container->get('webform_yuboto.api_service');
    $instance->tokenManager = $container->get('webform.token_manager');
    $instance->moduleHandler = $container->get('module_handler');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getSummary(): array {
    $fields = $this->getWebform()->getElementsInitializedAndFlattened();

    // Phone field summary.
    $phone_field = $this->configuration['phone_field'] ?? '';
    if (empty($phone_field)) {
      $phone_summary = $this->t('<strong>Phone</strong>: <em>No field selected</em>');
    }
    else {
      $phone_summary = $this->t('Phone: @field', ['@field' => $phone_field]);
      if (!empty($fields[$phone_field]['#title'])) {
        $phone_summary = $this->t(
          '<strong>Phone</strong>: @field (@title)', [
            '@field' => $phone_field,
            '@title' => $fields[$phone_field]['#title'],
          ]
        );
      }
    }

    // Message summary.
    $message = $this->configuration['message'] ?? '';
    if (empty($message)) {
      $message_summary = $this->t('<strong>Message</strong>: <em>Using default message</em>');
    }
    else {
      $message_summary = $this->t('<strong>Message</strong>: @message', ['@message' => mb_substr($message, 0, 50) . (mb_strlen($message) > 50 ? '...' : '')]);
    }

    // Sender summary.
    $sender = $this->configuration['sender'] ?? '';
    if (empty($sender)) {
      $sender_summary = $this->t('<strong>Sender</strong>: <em>Using default sender</em>');
    }
    else {
      $sender_summary = $this->t('<strong>Sender</strong>: <em>@sender</em>', ['@sender' => $sender]);
    }

    $markup = "{$phone_summary}<br>{$message_summary}<br>{$sender_summary}";
    return [
      '#markup' => $markup,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'phone_field' => '',
      'message' => '',
      'sender' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    // Get default values from module configuration for form display.
    $config = $this->configFactory->get('webform_yuboto.settings');

    $form['yuboto'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Yuboto SMS settings'),
      '#attributes' => ['id' => 'webform-yuboto-handler-settings'],
      '#tree' => TRUE,
    ];

    $fields = $this->getWebform()->getElementsInitializedAndFlattened();
    $phone_options = [];
    foreach ($fields as $field_name => $field) {
      if (in_array($field['#type'], [
        'tel',
        'textfield',
        'webform_telephone',
      ])) {
        $phone_options[$field_name] = $field['#title'];
      }
    }

    // Phone field selection.
    $default_phone = $this->configuration['phone_field'] ?? '';
    if (empty($default_phone) && count($phone_options) == 1) {
      $default_phone = key($phone_options);
    }

    $form['yuboto']['phone_field'] = [
      '#type' => 'webform_select_other',
      '#title' => $this->t('Phone number field'),
      '#required' => TRUE,
      '#default_value' => $default_phone,
      '#options' => $phone_options,
      '#empty_option' => $this->t('- Select a phone field -'),
      '#description' => $this->t('Select the phone number field from the form submission.'),
    ];

    // Message configuration.
    $form['yuboto']['message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('SMS Message'),
      '#default_value' => $this->configuration['message'] ?? $config->get('default_message') ?: 'Thank you for your submission.',
      '#description' => $this->t('Enter the SMS message to send. If empty, the default message will be used.'),
      '#rows' => 3,
    ];

    // Sender configuration.
    $form['yuboto']['sender'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Sender Name'),
      '#default_value' => $this->configuration['sender'] ?? $config->get('default_sender') ?: '[site:name]',
      '#description' => $this->t('Enter the sender name for the SMS. If empty, the default sender will be used.'),
    ];

    // Token help.
    $form['yuboto']['token_help'] = [
      '#theme' => 'token_tree_link',
      '#token_types' => ['webform', 'webform_submission', 'site'],
      '#global_types' => TRUE,
      '#click_insert' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);
    $values = $form_state->getValues();
    $yuboto_values = $values['yuboto'] ?? [];

    $this->configuration['phone_field'] = $yuboto_values['phone_field'] ?? '';
    $this->configuration['message'] = $yuboto_values['message'] ?? '';
    $this->configuration['sender'] = $yuboto_values['sender'] ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(WebformSubmissionInterface $webform_submission, $update = TRUE): void {
    // Only process new submissions, not updates.
    if ($update) {
      return;
    }

    $skip_handler = FALSE;
    // Allow other modules to skip the handler execution entirely.
    $this->moduleHandler->invokeAll(
      'webform_yuboto_skip_handler', [
        &$skip_handler,
        $webform_submission,
        $this,
      ]
    );

    if ($skip_handler) {
      if ($this->yubotoApi->isDebugEnabled()) {
        $this->getLogger('webform_yuboto')
          ->notice(
            'Yuboto handler execution skipped by hook for submission @sid', [
              '@sid' => $webform_submission->id(),
            ]
          );
      }
      return;
    }

    $skip_sms = FALSE;
    // Allow other modules to skip SMS sending.
    $this->moduleHandler->invokeAll(
      'webform_yuboto_skip_sms', [
        &$skip_sms,
        $webform_submission,
        $this,
      ]
    );

    if ($skip_sms) {
      if ($this->yubotoApi->isDebugEnabled()) {
        $this->getLogger('webform_yuboto')
          ->notice(
            'SMS sending skipped by hook for submission @sid', [
              '@sid' => $webform_submission->id(),
            ]
          );
      }
      return;
    }

    // Ensure all tokens are replaced (or cleared) before any API call.
    $token_options = ['clear' => TRUE];

    $configuration = $this->tokenManager->replace($this->configuration, $webform_submission, $token_options);
    $submission_data = $webform_submission->toArray(TRUE);

    // Get phone number from the specified field OR from a token-derived value.
    $phone_field = (string) ($configuration['phone_field'] ?? '');
    $phone_number = '';

    // If a custom value contains tokens, resolve them too
    // (e.g. "[webform_submission:values:phone]").
    if ($phone_field !== '') {
      $phone_field = (string) $this->tokenManager->replace($phone_field, $webform_submission, $token_options);
    }

    // First, treat it as a field key when it matches submission data.
    if ($phone_field !== '' && isset($submission_data['data'][$phone_field])) {
      $phone_number = (string) $submission_data['data'][$phone_field];
    }
    // Otherwise, treat it as a literal phone number if it contains digits.
    elseif ($phone_field !== '' && preg_match('/\d/', $phone_field)) {
      $phone_number = $phone_field;
    }

    if ($phone_number === '') {
      $this->getLogger('webform_yuboto')->warning(
        'No phone number found in field @field for submission @sid', [
          '@field' => $phone_field,
          '@sid' => $webform_submission->id(),
        ]
      );
      return;
    }

    // Prepare message and sender with token replacement.
    $global_yuboto_config = $this->configFactory->get('webform_yuboto.settings')
      ->getRawData();
    $message = $configuration['message'] !== '' ? $configuration['message'] : $global_yuboto_config['default_message'];
    $sender = $configuration['sender'] !== '' ? $configuration['sender'] : $global_yuboto_config['default_sender'];

    // Additional token replacement for complex tokens.
    $message = (string) $this->tokenManager->replace($message, $webform_submission, $token_options);
    $sender = (string) $this->tokenManager->replace($sender, $webform_submission, $token_options);

    try {
      $result = $this->yubotoApi->sendSms($phone_number, $message, $sender);
      // An error happened. Do not log it, it is already logged by the API.
      if (isset($result['status']['error'])) {
        return;
      }
    }
    catch (GuzzleException $e) {
      $this->getLogger('webform_yuboto')->error($e->getMessage());
    }
  }

}
